// LibTiePieHW.pas - Unit for using libtiepie-hw with Pascal.
//
// (c) TiePie engineering 2025
//
// For API documentation see: http://api.tiepie.com/libtiepie-hw
//
// Website: https://www.tiepie.com

unit LibTiePieHW;

{$ifdef FPC}
  {$MODE DELPHI}
{$endif FPC}

interface

uses
  {$ifdef MSWINDOWS}
  Windows,
  Messages,
  {$endif MSWINDOWS}
  Types;

const
  TIEPIE_HW_VERSION_MAJOR = 1;
  TIEPIE_HW_VERSION_MINOR = 5;
  TIEPIE_HW_VERSION_PATCH = 0;
  TIEPIE_HW_VERSION_NUMBER = '1.5.0';

  TIEPIE_HW_VERSION = '1.5.0';

  TIEPIE_HW_HANDLE_INVALID = 0; // Invalid handle

  TIEPIE_HW_INTERFACE_DEVICE = $0000000000000001; // Device
  TIEPIE_HW_INTERFACE_OSCILLOSCOPE = $0000000000000002; // Oscilloscope
  TIEPIE_HW_INTERFACE_GENERATOR = $0000000000000004; // Generator
  TIEPIE_HW_INTERFACE_SERVER = $0000000000000010; // Server
  TIEPIE_HW_INTERFACE_COUNT = 4; // Number of devices

  TIEPIE_HW_DEVICETYPE_OSCILLOSCOPE = $00000001; // Oscilloscope
  TIEPIE_HW_DEVICETYPE_GENERATOR = $00000002; // Generator
  TIEPIE_HW_DEVICETYPE_COUNT = 2; // Number of device types

  TIEPIE_HW_CONNECTORTYPE_UNKNOWN = $00000000; // Unknown connector
  TIEPIE_HW_CONNECTORTYPE_BNC = $00000001; // BNC connector
  TIEPIE_HW_CONNECTORTYPE_BANANA = $00000002; // Banana connector
  TIEPIE_HW_CONNECTORTYPE_POWERPLUG = $00000004; // Mains power plug connector
  TIEPIE_HW_CONNECTORTYPE_COUNT = 3; // Number of connector types
  TIEPIE_HW_CONNECTORTYPE_MASK = (TIEPIE_HW_CONNECTORTYPE_BNC or TIEPIE_HW_CONNECTORTYPE_BANANA or TIEPIE_HW_CONNECTORTYPE_POWERPLUG); // Connector mask type

  TIEPIE_HW_DATARAWTYPE_UNKNOWN = $00000000; // unknown raw data type
  TIEPIE_HW_DATARAWTYPE_INT8 = $00000001; // int8_t
  TIEPIE_HW_DATARAWTYPE_INT16 = $00000002; // int16_t
  TIEPIE_HW_DATARAWTYPE_INT32 = $00000004; // int32_t
  TIEPIE_HW_DATARAWTYPE_INT64 = $00000008; // int64_t
  TIEPIE_HW_DATARAWTYPE_UINT8 = $00000010; // uint8_t
  TIEPIE_HW_DATARAWTYPE_UINT16 = $00000020; // uint16_t
  TIEPIE_HW_DATARAWTYPE_UINT32 = $00000040; // uint32_t
  TIEPIE_HW_DATARAWTYPE_UINT64 = $00000080; // uint64_t
  TIEPIE_HW_DATARAWTYPE_FLOAT32 = $00000100; // float
  TIEPIE_HW_DATARAWTYPE_FLOAT64 = $00000200; // double
  TIEPIE_HW_DATARAWTYPE_COUNT = 10; // Number of raw data types
  TIEPIE_HW_DATARAWTYPE_MASK_INT = (TIEPIE_HW_DATARAWTYPE_INT8 or TIEPIE_HW_DATARAWTYPE_INT16 or TIEPIE_HW_DATARAWTYPE_INT32 or TIEPIE_HW_DATARAWTYPE_INT64); // Integer raw data types
  TIEPIE_HW_DATARAWTYPE_MASK_UINT = (TIEPIE_HW_DATARAWTYPE_UINT8 or TIEPIE_HW_DATARAWTYPE_UINT16 or TIEPIE_HW_DATARAWTYPE_UINT32 or TIEPIE_HW_DATARAWTYPE_UINT64); // Unsigned integer raw data types
  TIEPIE_HW_DATARAWTYPE_MASK_FLOAT = (TIEPIE_HW_DATARAWTYPE_FLOAT32 or TIEPIE_HW_DATARAWTYPE_FLOAT64 ); // Float raw data types
  TIEPIE_HW_DATARAWTYPE_MASK_FIXED = (TIEPIE_HW_DATARAWTYPE_MASK_INT or TIEPIE_HW_DATARAWTYPE_MASK_UINT); // Fixed raw data types

  TIEPIE_HW_TRISTATE_UNDEFINED = 0; // Undefined
  TIEPIE_HW_TRISTATE_FALSE = 1; // False
  TIEPIE_HW_TRISTATE_TRUE = 2; // True

  TIEPIE_HW_TRIGGERIO_INDEX_INVALID = $FFFF; // Trigger IO index invalid

  TIEPIE_HW_STRING_LENGTH_NULL_TERMINATED = $FFFFFFFF; // String length nul terminated

  TIEPIE_HW_SERVER_STATUS_DISCONNECTED = 0; // Server status: disconnected
  TIEPIE_HW_SERVER_STATUS_CONNECTING = 1; // Server status: connecting
  TIEPIE_HW_SERVER_STATUS_CONNECTED = 2; // Server status: connected
  TIEPIE_HW_SERVER_STATUS_DISCONNECTING = 3; // Server status: disconnecting
  TIEPIE_HW_SERVER_ERROR_NONE = 0; // Server error: none
  TIEPIE_HW_SERVER_ERROR_UNKNOWN = 1; // Server error: unknown
  TIEPIE_HW_SERVER_ERROR_CONNECTIONREFUSED = 2; // Server error: connection refused
  TIEPIE_HW_SERVER_ERROR_NETWORKUNREACHABLE = 3; // Server error: network unreachable
  TIEPIE_HW_SERVER_ERROR_TIMEDOUT = 4; // Server error: timed out
  TIEPIE_HW_SERVER_ERROR_HOSTNAMELOOKUPFAILED = 5; // Server error: host name lookup failed

  TIEPIE_HW_RANGEINDEX_AUTO = $FFFFFFFF; // Auto ranging

  TIEPIE_HW_POINTER_ARRAY_MAX_LENGTH = 256; // Pointer array maximum length

  TIEPIE_HW_ARMN_COUNT = 3; // Number of auto resolution modes

  TIEPIE_HW_ARMB_DISABLED = 0; // Bit number for Disabled
  TIEPIE_HW_ARMB_NATIVEONLY = 1; // Bit number for Native only
  TIEPIE_HW_ARMB_ALL = 2; // Bit number for All

  TIEPIE_HW_ARM_UNKNOWN = 0; // Unknown/invalid mode
  TIEPIE_HW_ARM_DISABLED = (1 shl TIEPIE_HW_ARMB_DISABLED); // Resolution does not automatically change.
  TIEPIE_HW_ARM_NATIVEONLY = (1 shl TIEPIE_HW_ARMB_NATIVEONLY); // Highest possible native resolution for the current sample rate is used.
  TIEPIE_HW_ARM_ALL = (1 shl TIEPIE_HW_ARMB_ALL); // Highest possible native or enhanced resolution for the current sample rate is used.

  TIEPIE_HW_ARMM_NONE = 0; // None
  TIEPIE_HW_ARMM_ALL = ((1 shl TIEPIE_HW_ARMN_COUNT) - 1); // All
  TIEPIE_HW_ARMM_ENABLED = (TIEPIE_HW_ARMM_ALL and not TIEPIE_HW_ARM_DISABLED); // Enabled

  TIEPIE_HW_CKN_COUNT = 5; // Number of couplings

  TIEPIE_HW_CKB_DCV = 0; // Bit number for input coupling Volt DC
  TIEPIE_HW_CKB_ACV = 1; // Bit number for input coupling Volt AC
  TIEPIE_HW_CKB_DCA = 2; // Bit number for input coupling Ampere DC
  TIEPIE_HW_CKB_ACA = 3; // Bit number for input coupling Ampere AC
  TIEPIE_HW_CKB_OHM = 4; // Bit number for input coupling Ohm

  TIEPIE_HW_CK_UNKNOWN = 0; // Unknown/invalid coupling
  TIEPIE_HW_CK_DCV = (1 shl TIEPIE_HW_CKB_DCV); // Input coupling Volt DC
  TIEPIE_HW_CK_ACV = (1 shl TIEPIE_HW_CKB_ACV); // Input coupling Volt AC
  TIEPIE_HW_CK_DCA = (1 shl TIEPIE_HW_CKB_DCA); // Input coupling Ampere DC
  TIEPIE_HW_CK_ACA = (1 shl TIEPIE_HW_CKB_ACA); // Input coupling Ampere AC
  TIEPIE_HW_CK_OHM = (1 shl TIEPIE_HW_CKB_OHM); // Input coupling Ohm

  TIEPIE_HW_CKM_NONE = 0; // None
  TIEPIE_HW_CKM_V = (TIEPIE_HW_CK_DCV or TIEPIE_HW_CK_ACV); // Input couplings that allow measuring Volt
  TIEPIE_HW_CKM_A = (TIEPIE_HW_CK_DCA or TIEPIE_HW_CK_ACA); // Input couplings that allow measuring Ampere
  TIEPIE_HW_CKM_OHM = (TIEPIE_HW_CK_OHM); // Input couplings that allow measuring Ohm
  TIEPIE_HW_CKM_ASYMMETRICRANGE = (TIEPIE_HW_CKM_OHM); // Input couplings that have an asymmetric range: 0 to +Range
  TIEPIE_HW_CKM_SYMMETRICRANGE = (TIEPIE_HW_CKM_V or TIEPIE_HW_CKM_A); // Input couplings that have a symmetric range: -Range to +Range
  TIEPIE_HW_CKM_DC = (TIEPIE_HW_CK_DCV or TIEPIE_HW_CK_DCA or TIEPIE_HW_CK_OHM); // input couplings that have DC coupling
  TIEPIE_HW_CKM_AC = (TIEPIE_HW_CK_ACV or TIEPIE_HW_CK_ACA); // input couplings that have AC coupling

  TIEPIE_HW_CON_COUNT = 3; // Number of clock output types

  TIEPIE_HW_COB_DISABLED = 0; // Bit number for No clock output
  TIEPIE_HW_COB_SAMPLE = 1; // Bit number for Sample clock
  TIEPIE_HW_COB_FIXED = 2; // Bit number for Fixed clock

  TIEPIE_HW_CO_DISABLED = (1 shl TIEPIE_HW_COB_DISABLED); // No clock output
  TIEPIE_HW_CO_SAMPLE = (1 shl TIEPIE_HW_COB_SAMPLE); // Sample clock
  TIEPIE_HW_CO_FIXED = (1 shl TIEPIE_HW_COB_FIXED); // Fixed clock

  TIEPIE_HW_COM_NONE = 0; // None
  TIEPIE_HW_COM_ALL = ((1 shl TIEPIE_HW_CON_COUNT) - 1); // All clock output types
  TIEPIE_HW_COM_ENABLED = (TIEPIE_HW_COM_ALL and not TIEPIE_HW_CO_DISABLED); // All clock output types with Enabled output
  TIEPIE_HW_COM_FREQUENCY = (TIEPIE_HW_CO_FIXED); // All clock output types with Frequency output

  TIEPIE_HW_CSN_COUNT = 2; // Number of clock sources

  TIEPIE_HW_CSB_EXTERNAL = 0; // Bit number for External clock
  TIEPIE_HW_CSB_INTERNAL = 1; // Bit number for Internal clock

  TIEPIE_HW_CS_EXTERNAL = (1 shl TIEPIE_HW_CSB_EXTERNAL); // Clock source External clock
  TIEPIE_HW_CS_INTERNAL = (1 shl TIEPIE_HW_CSB_INTERNAL); // Clock source Internal clock

  TIEPIE_HW_CSM_NONE = 0; // None
  TIEPIE_HW_CSM_ALL = ((1 shl TIEPIE_HW_CSN_COUNT) - 1); // All clock sources
  TIEPIE_HW_CSM_FREQUENCY = (TIEPIE_HW_CS_EXTERNAL); // All External clock sources

  TIEPIE_HW_FMN_COUNT = 2; // Number of frequency modes

  TIEPIE_HW_FMB_SIGNALFREQUENCY = 0; // Bit number for Frequency mode: Signal frequency
  TIEPIE_HW_FMB_SAMPLERATE = 1; // Bit number for Frequency mode: Sampling rate

  TIEPIE_HW_FM_UNKNOWN = $00000000; // Unknown Frequency mode
  TIEPIE_HW_FM_SIGNALFREQUENCY = (1 shl TIEPIE_HW_FMB_SIGNALFREQUENCY); // Frequency mode: Signal frequency
  TIEPIE_HW_FM_SAMPLERATE = (1 shl TIEPIE_HW_FMB_SAMPLERATE); // Frequency mode: Sampling rate

  TIEPIE_HW_FMM_NONE = $00000000; // None
  TIEPIE_HW_FMM_ALL = ((1 shl TIEPIE_HW_FMN_COUNT) - 1); // All

  TIEPIE_HW_GMN_COUNT = 12; // Number of generator modes

  TIEPIE_HW_GMB_CONTINUOUS = 0; // Bit number for generator mode Continuous
  TIEPIE_HW_GMB_BURST_COUNT = 1; // Bit number for generator mode Burst count
  TIEPIE_HW_GMB_GATED_PERIODS = 2; // Bit number for generator mode Gated periods
  TIEPIE_HW_GMB_GATED = 3; // Bit number for generator mode Gated
  TIEPIE_HW_GMB_GATED_PERIOD_START = 4; // Bit number for generator mode Period start
  TIEPIE_HW_GMB_GATED_PERIOD_FINISH = 5; // Bit number for generator mode Period finish
  TIEPIE_HW_GMB_GATED_RUN = 6; // Bit number for generator mode Gated run
  TIEPIE_HW_GMB_GATED_RUN_OUTPUT = 7; // Bit number for generator mode Gated run output
  TIEPIE_HW_GMB_BURST_SAMPLE_COUNT = 8; // Bit number for generator mode Sample count
  TIEPIE_HW_GMB_BURST_SAMPLE_COUNT_OUTPUT = 9; // Bit number for generator mode Sample count output
  TIEPIE_HW_GMB_BURST_SEGMENT_COUNT = 10; // Bit number for generator mode Segment count
  TIEPIE_HW_GMB_BURST_SEGMENT_COUNT_OUTPUT = 11; // Bit number for generator mode Segment count output

  TIEPIE_HW_GM_UNKNOWN = 0; // Generator mode: unknown
  TIEPIE_HW_GM_CONTINUOUS = (1 shl TIEPIE_HW_GMB_CONTINUOUS); // Generator mode: Continuous
  TIEPIE_HW_GM_BURST_COUNT = (1 shl TIEPIE_HW_GMB_BURST_COUNT); // Generator mode: Burst count
  TIEPIE_HW_GM_GATED_PERIODS = (1 shl TIEPIE_HW_GMB_GATED_PERIODS); // Generator mode: Gated periods
  TIEPIE_HW_GM_GATED = (1 shl TIEPIE_HW_GMB_GATED); // Generator mode: Gated
  TIEPIE_HW_GM_GATED_PERIOD_START = (1 shl TIEPIE_HW_GMB_GATED_PERIOD_START); // Generator mode: Period start
  TIEPIE_HW_GM_GATED_PERIOD_FINISH = (1 shl TIEPIE_HW_GMB_GATED_PERIOD_FINISH); // Generator mode: Period finish
  TIEPIE_HW_GM_GATED_RUN = (1 shl TIEPIE_HW_GMB_GATED_RUN); // Generator mode: Gated run
  TIEPIE_HW_GM_GATED_RUN_OUTPUT = (1 shl TIEPIE_HW_GMB_GATED_RUN_OUTPUT); // Generator mode: Gated run output
  TIEPIE_HW_GM_BURST_SAMPLE_COUNT = (1 shl TIEPIE_HW_GMB_BURST_SAMPLE_COUNT); // Generator mode: Sample count
  TIEPIE_HW_GM_BURST_SAMPLE_COUNT_OUTPUT = (1 shl TIEPIE_HW_GMB_BURST_SAMPLE_COUNT_OUTPUT); // Generator mode: Sample count output
  TIEPIE_HW_GM_BURST_SEGMENT_COUNT = (1 shl TIEPIE_HW_GMB_BURST_SEGMENT_COUNT); // Generator mode: Segment count
  TIEPIE_HW_GM_BURST_SEGMENT_COUNT_OUTPUT = (1 shl TIEPIE_HW_GMB_BURST_SEGMENT_COUNT_OUTPUT); // Generator mode: Segment count output

  TIEPIE_HW_GMM_NONE = 0; // None
  TIEPIE_HW_GMM_BURST_COUNT = (TIEPIE_HW_GM_BURST_COUNT); // Generator modes that involve Burst count
  TIEPIE_HW_GMM_GATED = (TIEPIE_HW_GM_GATED_PERIODS or TIEPIE_HW_GM_GATED or TIEPIE_HW_GM_GATED_PERIOD_START or TIEPIE_HW_GM_GATED_PERIOD_FINISH or TIEPIE_HW_GM_GATED_RUN or TIEPIE_HW_GM_GATED_RUN_OUTPUT); // Generator modes that involve Gated mode
  TIEPIE_HW_GMM_BURST_SAMPLE_COUNT = (TIEPIE_HW_GM_BURST_SAMPLE_COUNT or TIEPIE_HW_GM_BURST_SAMPLE_COUNT_OUTPUT); // Generator modes that involve Burst sample count
  TIEPIE_HW_GMM_BURST_SEGMENT_COUNT = (TIEPIE_HW_GM_BURST_SEGMENT_COUNT or TIEPIE_HW_GM_BURST_SEGMENT_COUNT_OUTPUT); // Generator modes that involve Burst segment count
  TIEPIE_HW_GMM_BURST = (TIEPIE_HW_GMM_BURST_COUNT or TIEPIE_HW_GMM_BURST_SAMPLE_COUNT or TIEPIE_HW_GMM_BURST_SEGMENT_COUNT); // Generator modes that involve Burst xx count
  TIEPIE_HW_GMM_REQUIRE_TRIGGER = (TIEPIE_HW_GMM_GATED or TIEPIE_HW_GMM_BURST_SAMPLE_COUNT or TIEPIE_HW_GMM_BURST_SEGMENT_COUNT); // Generator modes that require an enabeld trigger input.
  TIEPIE_HW_GMM_ALL = ((1 shl TIEPIE_HW_GMN_COUNT) - 1); // All generator modes
  TIEPIE_HW_GMM_SIGNALFREQUENCY = (TIEPIE_HW_GMM_ALL and not TIEPIE_HW_GMM_BURST_SAMPLE_COUNT); // Supported generator modes when frequency mode is signal frequency.
  TIEPIE_HW_GMM_SAMPLERATE = (TIEPIE_HW_GMM_ALL); // Supported generator modes when frequency mode is sample rate.
  TIEPIE_HW_GMM_SINE = (TIEPIE_HW_GMM_SIGNALFREQUENCY); // Supported generator modes when signal type is sine.
  TIEPIE_HW_GMM_TRIANGLE = (TIEPIE_HW_GMM_SIGNALFREQUENCY); // Supported generator modes when signal type is triangle.
  TIEPIE_HW_GMM_SQUARE = (TIEPIE_HW_GMM_SIGNALFREQUENCY); // Supported generator modes when signal type is square.
  TIEPIE_HW_GMM_DC = (TIEPIE_HW_GM_CONTINUOUS); // Supported generator modes when signal type is TIEPIE_HW_DC.
  TIEPIE_HW_GMM_NOISE = (TIEPIE_HW_GM_CONTINUOUS or TIEPIE_HW_GM_GATED); // Supported generator modes when signal type is noise.
  TIEPIE_HW_GMM_ARBITRARY = (TIEPIE_HW_GMM_SIGNALFREQUENCY or TIEPIE_HW_GMM_SAMPLERATE); // Supported generator modes when signal type is arbitrary.
  TIEPIE_HW_GMM_PULSE = (TIEPIE_HW_GMM_SIGNALFREQUENCY and not TIEPIE_HW_GMM_BURST_SEGMENT_COUNT); // Supported generator modes when signal type is pulse.

  TIEPIE_HW_GSN_COUNT = 4; // The number of generator status flags.

  TIEPIE_HW_GSB_STOPPED = 0; // Bit number for signal generation stopped.
  TIEPIE_HW_GSB_RUNNING = 1; // Bit number for signal generation running.
  TIEPIE_HW_GSB_BURSTACTIVE = 2; // Bit number for operating in burst mode.
  TIEPIE_HW_GSB_WAITING = 3; // Bit number for waiting for a burst to be started.

  TIEPIE_HW_GS_STOPPED = (1 shl TIEPIE_HW_GSB_STOPPED); // The signal generation is stopped.
  TIEPIE_HW_GS_RUNNING = (1 shl TIEPIE_HW_GSB_RUNNING); // The signal generation is running.
  TIEPIE_HW_GS_BURSTACTIVE = (1 shl TIEPIE_HW_GSB_BURSTACTIVE); // The generator is operating in burst mode.
  TIEPIE_HW_GS_WAITING = (1 shl TIEPIE_HW_GSB_WAITING); // The generator is waiting for a burst to be started.

  TIEPIE_HW_GSM_NONE = 0; // None
  TIEPIE_HW_GSM_ALL = ((1 shl TIEPIE_HW_GSN_COUNT) - 1); // All signal generation status flags of a generator

  TIEPIE_HW_MMN_COUNT = 2; // Number of measure modes

  TIEPIE_HW_MMB_STREAM = 0; // Bit number for Streaming mode
  TIEPIE_HW_MMB_BLOCK = 1; // Bit number for Block mode

  TIEPIE_HW_MMM_NONE = 0; // None
  TIEPIE_HW_MMM_ALL = ((1 shl TIEPIE_HW_MMN_COUNT) - 1); // All measure mode settings of an oscilloscope

  TIEPIE_HW_MM_UNKNOWN = 0; // Unknown/invalid mode
  TIEPIE_HW_MM_STREAM = (1 shl TIEPIE_HW_MMB_STREAM); // Streaming mode
  TIEPIE_HW_MM_BLOCK = (1 shl TIEPIE_HW_MMB_BLOCK ); // Block mode

  TIEPIE_HW_STN_COUNT = 7; // Number of signal types

  TIEPIE_HW_STB_SINE = 0; // Bit number for signal type Sine
  TIEPIE_HW_STB_TRIANGLE = 1; // Bit number for signal type Triangle
  TIEPIE_HW_STB_SQUARE = 2; // Bit number for signal type Square
  TIEPIE_HW_STB_DC = 3; // Bit number for signal type DC
  TIEPIE_HW_STB_NOISE = 4; // Bit number for signal type Noise
  TIEPIE_HW_STB_ARBITRARY = 5; // Bit number for signal type Arbitrary
  TIEPIE_HW_STB_PULSE = 6; // Bit number for signal type Pulse

  TIEPIE_HW_ST_UNKNOWN = 0; // Signal type unknown/invalid
  TIEPIE_HW_ST_SINE = (1 shl TIEPIE_HW_STB_SINE); // Signal type Sine
  TIEPIE_HW_ST_TRIANGLE = (1 shl TIEPIE_HW_STB_TRIANGLE); // Signal type Triangle
  TIEPIE_HW_ST_SQUARE = (1 shl TIEPIE_HW_STB_SQUARE); // Signal type Square
  TIEPIE_HW_ST_DC = (1 shl TIEPIE_HW_STB_DC); // Signal type DC
  TIEPIE_HW_ST_NOISE = (1 shl TIEPIE_HW_STB_NOISE); // Signal type Noise
  TIEPIE_HW_ST_ARBITRARY = (1 shl TIEPIE_HW_STB_ARBITRARY); // Signal type Arbitrary
  TIEPIE_HW_ST_PULSE = (1 shl TIEPIE_HW_STB_PULSE); // Signal type Pulse

  TIEPIE_HW_STM_NONE = 0; // None
  TIEPIE_HW_STM_AMPLITUDE = (TIEPIE_HW_ST_SINE or TIEPIE_HW_ST_TRIANGLE or TIEPIE_HW_ST_SQUARE or TIEPIE_HW_ST_NOISE or TIEPIE_HW_ST_ARBITRARY or TIEPIE_HW_ST_PULSE); // Signal types that support setting amplitude
  TIEPIE_HW_STM_OFFSET = (TIEPIE_HW_ST_SINE or TIEPIE_HW_ST_TRIANGLE or TIEPIE_HW_ST_SQUARE or TIEPIE_HW_ST_DC or TIEPIE_HW_ST_NOISE or TIEPIE_HW_ST_ARBITRARY or TIEPIE_HW_ST_PULSE); // Signal types that support setting offset
  TIEPIE_HW_STM_FREQUENCY = (TIEPIE_HW_ST_SINE or TIEPIE_HW_ST_TRIANGLE or TIEPIE_HW_ST_SQUARE or TIEPIE_HW_ST_NOISE or TIEPIE_HW_ST_ARBITRARY or TIEPIE_HW_ST_PULSE); // Signal types that support setting frequency
  TIEPIE_HW_STM_PHASE = (TIEPIE_HW_ST_SINE or TIEPIE_HW_ST_TRIANGLE or TIEPIE_HW_ST_SQUARE or TIEPIE_HW_ST_ARBITRARY or TIEPIE_HW_ST_PULSE); // Signal types that support setting phase
  TIEPIE_HW_STM_SYMMETRY = (TIEPIE_HW_ST_SINE or TIEPIE_HW_ST_TRIANGLE or TIEPIE_HW_ST_SQUARE ); // Signal types that support setting symmetry
  TIEPIE_HW_STM_WIDTH = ( TIEPIE_HW_ST_PULSE); // Signal types that support setting width
  TIEPIE_HW_STM_LEADINGEDGETIME = ( TIEPIE_HW_ST_PULSE); // Signal types that support setting leading edge time
  TIEPIE_HW_STM_TRAILINGEDGETIME = ( TIEPIE_HW_ST_PULSE); // Signal types that support setting trailing edge time
  TIEPIE_HW_STM_DATALENGTH = ( TIEPIE_HW_ST_ARBITRARY ); // Signal types that support setting data length
  TIEPIE_HW_STM_DATA = ( TIEPIE_HW_ST_ARBITRARY ); // Signal types that support setting data
  TIEPIE_HW_STM_EDGETIME = (TIEPIE_HW_STM_LEADINGEDGETIME and TIEPIE_HW_STM_TRAILINGEDGETIME); // Edge time

  TIEPIE_HW_TCN_COUNT = 5; // Number of trigger conditions

  TIEPIE_HW_TCB_NONE = 0; // Bit number for no trigger condition
  TIEPIE_HW_TCB_SMALLER = 1; // Bit number for trigger condition Smaller
  TIEPIE_HW_TCB_LARGER = 2; // Bit number for trigger condition Larger
  TIEPIE_HW_TCB_INSIDE = 3; // Bit number for trigger condition Inside
  TIEPIE_HW_TCB_OUTSIDE = 4; // Bit number for trigger condition Outside

  TIEPIE_HW_TC_UNKNOWN = 0; // Unknown/invalid trigger condition
  TIEPIE_HW_TC_NONE = (1 shl TIEPIE_HW_TCB_NONE); // No trigger condition
  TIEPIE_HW_TC_SMALLER = (1 shl TIEPIE_HW_TCB_SMALLER); // Trigger condition Smaller
  TIEPIE_HW_TC_LARGER = (1 shl TIEPIE_HW_TCB_LARGER); // Trigger condition Larger
  TIEPIE_HW_TC_INSIDE = (1 shl TIEPIE_HW_TCB_INSIDE); // Trigger condition Inside
  TIEPIE_HW_TC_OUTSIDE = (1 shl TIEPIE_HW_TCB_OUTSIDE); // Trigger condition Outside

  TIEPIE_HW_TCM_NONE = 0; // No conditions
  TIEPIE_HW_TCM_ALL = ((1 shl TIEPIE_HW_TCN_COUNT) - 1); // All conditions
  TIEPIE_HW_TCM_ENABLED = (TIEPIE_HW_TCM_ALL and not TIEPIE_HW_TC_NONE); // All conditions except #TIEPIE_HW_TC_NONE.

  TIEPIE_HW_DN_MAIN = 0; // The device itself.
  TIEPIE_HW_DN_SUB_FIRST = 1; // First sub device in a combined device.
  TIEPIE_HW_DN_SUB_SECOND = 2; // Second sub device in a combined device.

  TIEPIE_HW_PGID_OSCILLOSCOPE = 1; // Oscilloscope
  TIEPIE_HW_PGID_GENERATOR = 2; // Generator
  TIEPIE_HW_PGID_EXTERNAL_DSUB = 3; // External D-sub

  TIEPIE_HW_SGID_MAIN = 0; // The oscilloscope or function generator itself.
  TIEPIE_HW_SGID_CHANNEL1 = 1; // Channel 1
  TIEPIE_HW_SGID_CHANNEL2 = 2; // Channel 2
  TIEPIE_HW_SGID_PIN1 = 1; // Pin 1
  TIEPIE_HW_SGID_PIN2 = 2; // Pin 2
  TIEPIE_HW_SGID_PIN3 = 3; // Pin 3

  TIEPIE_HW_FID_SCP_TRIGGERED = 0; // Triggered
  TIEPIE_HW_FID_GEN_START = 0; // Generator Start
  TIEPIE_HW_FID_GEN_STOP = 1; // Generator Stop
  TIEPIE_HW_FID_GEN_NEW_PERIOD = 2; // Generator New Period
  TIEPIE_HW_FID_EXT_TRIGGERED = 0; // External Triggered

  TIEPIE_HW_TIOID_SHIFT_PGID = 20; // Bit Shift TIEPIE_HW_PGID
  TIEPIE_HW_TIOID_SHIFT_DN = 24; // Bit Shift TIEPIE_HW_DN
  TIEPIE_HW_TIOID_SHIFT_SGID = 8; // Bit Shift TIEPIE_HW_SGID
  TIEPIE_HW_TIOID_SHIFT_FID = 0; // Bit Shift TIEPIE_HW_FID

  TIEPIE_HW_TIID_INVALID = 0; // Trigger Input TIEPIE_HW_ID invalid
  TIEPIE_HW_TIID_EXT1 = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_EXTERNAL_DSUB) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_PIN1) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_EXT_TRIGGERED) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Input TIEPIE_HW_ID TIEPIE_HW_EXT1
  TIEPIE_HW_TIID_EXT2 = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_EXTERNAL_DSUB) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_PIN2) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_EXT_TRIGGERED) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Input TIEPIE_HW_ID TIEPIE_HW_EXT2
  TIEPIE_HW_TIID_EXT3 = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_EXTERNAL_DSUB) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_PIN3) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_EXT_TRIGGERED) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Input TIEPIE_HW_ID TIEPIE_HW_EXT3
  TIEPIE_HW_TIID_GENERATOR_START = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_GENERATOR) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_MAIN) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_GEN_START) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Input TIEPIE_HW_ID Generator Start
  TIEPIE_HW_TIID_GENERATOR_STOP = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_GENERATOR) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_MAIN) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_GEN_STOP) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Input TIEPIE_HW_ID Generator Stop
  TIEPIE_HW_TIID_GENERATOR_NEW_PERIOD = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_GENERATOR) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_MAIN) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_GEN_NEW_PERIOD) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Input TIEPIE_HW_ID Generator New Period

  TIEPIE_HW_TOID_INVALID = 0; // Trigger Output TIEPIE_HW_ID invalid
  TIEPIE_HW_TOID_EXT1 = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_EXTERNAL_DSUB) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_PIN1) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_EXT_TRIGGERED) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Output TIEPIE_HW_ID TIEPIE_HW_EXT1
  TIEPIE_HW_TOID_EXT2 = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_EXTERNAL_DSUB) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_PIN2) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_EXT_TRIGGERED) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Output TIEPIE_HW_ID TIEPIE_HW_EXT2
  TIEPIE_HW_TOID_EXT3 = ((TIEPIE_HW_DN_MAIN shl TIEPIE_HW_TIOID_SHIFT_DN) or ((TIEPIE_HW_PGID_EXTERNAL_DSUB) shl TIEPIE_HW_TIOID_SHIFT_PGID) or ((TIEPIE_HW_SGID_PIN3) shl TIEPIE_HW_TIOID_SHIFT_SGID) or ((TIEPIE_HW_FID_EXT_TRIGGERED) shl TIEPIE_HW_TIOID_SHIFT_FID)); // Trigger Output TIEPIE_HW_ID TIEPIE_HW_EXT3

  TIEPIE_HW_TKN_COUNT = 15; // Number of trigger kinds

  TIEPIE_HW_TKB_RISINGEDGE = 0; // Bit number for Rising edge
  TIEPIE_HW_TKB_FALLINGEDGE = 1; // Bit number for Falling edge
  TIEPIE_HW_TKB_INWINDOW = 2; // Bit number for Inside window
  TIEPIE_HW_TKB_OUTWINDOW = 3; // Bit number for Outside window
  TIEPIE_HW_TKB_ANYEDGE = 4; // Bit number for Any edge
  TIEPIE_HW_TKB_ENTERWINDOW = 5; // Bit number for Enter window
  TIEPIE_HW_TKB_EXITWINDOW = 6; // Bit number for Exit window
  TIEPIE_HW_TKB_PULSEWIDTHPOSITIVE = 7; // Bit number for Positive pulse width
  TIEPIE_HW_TKB_PULSEWIDTHNEGATIVE = 8; // Bit number for Negative pulse width
  TIEPIE_HW_TKB_PULSEWIDTHEITHER = 9; // Bit number for Positive or negative pulse width
  TIEPIE_HW_TKB_RUNTPULSEPOSITIVE = 10; // Bit number for Positive runt pulse
  TIEPIE_HW_TKB_RUNTPULSENEGATIVE = 11; // Bit number for Negative runt pulse
  TIEPIE_HW_TKB_RUNTPULSEEITHER = 12; // Bit number for Positive or negative runt pulse
  TIEPIE_HW_TKB_INTERVALRISING = 13; // Bit number for Interval (rising edge)
  TIEPIE_HW_TKB_INTERVALFALLING = 14; // Bit number for Interval (falling edge)

  TIEPIE_HW_TK_UNKNOWN = 0; // Unknown/invalid trigger kind
  TIEPIE_HW_TK_RISINGEDGE = (1 shl TIEPIE_HW_TKB_RISINGEDGE); // Rising edge
  TIEPIE_HW_TK_FALLINGEDGE = (1 shl TIEPIE_HW_TKB_FALLINGEDGE); // Falling edge
  TIEPIE_HW_TK_INWINDOW = (1 shl TIEPIE_HW_TKB_INWINDOW); // Inside window
  TIEPIE_HW_TK_OUTWINDOW = (1 shl TIEPIE_HW_TKB_OUTWINDOW); // Outside window
  TIEPIE_HW_TK_ANYEDGE = (1 shl TIEPIE_HW_TKB_ANYEDGE); // Any edge
  TIEPIE_HW_TK_ENTERWINDOW = (1 shl TIEPIE_HW_TKB_ENTERWINDOW); // Enter window
  TIEPIE_HW_TK_EXITWINDOW = (1 shl TIEPIE_HW_TKB_EXITWINDOW); // Exit window
  TIEPIE_HW_TK_PULSEWIDTHPOSITIVE = (1 shl TIEPIE_HW_TKB_PULSEWIDTHPOSITIVE); // Positive pulse width
  TIEPIE_HW_TK_PULSEWIDTHNEGATIVE = (1 shl TIEPIE_HW_TKB_PULSEWIDTHNEGATIVE); // Negative pulse width
  TIEPIE_HW_TK_PULSEWIDTHEITHER = (1 shl TIEPIE_HW_TKB_PULSEWIDTHEITHER); // Positive or negative pulse width
  TIEPIE_HW_TK_RUNTPULSEPOSITIVE = (1 shl TIEPIE_HW_TKB_RUNTPULSEPOSITIVE); // Positive runt pulse
  TIEPIE_HW_TK_RUNTPULSENEGATIVE = (1 shl TIEPIE_HW_TKB_RUNTPULSENEGATIVE); // Negative runt pulse
  TIEPIE_HW_TK_RUNTPULSEEITHER = (1 shl TIEPIE_HW_TKB_RUNTPULSEEITHER); // Positive or negative runt pulse
  TIEPIE_HW_TK_INTERVALRISING = (1 shl TIEPIE_HW_TKB_INTERVALRISING); // Interval (rising edge)
  TIEPIE_HW_TK_INTERVALFALLING = (1 shl TIEPIE_HW_TKB_INTERVALFALLING); // Interval (falling edge)

  TIEPIE_HW_TKM_NONE = 0; // No trigger kinds
  TIEPIE_HW_TKM_EDGE = (TIEPIE_HW_TK_RISINGEDGE or TIEPIE_HW_TK_FALLINGEDGE or TIEPIE_HW_TK_ANYEDGE); // All edge triggers
  TIEPIE_HW_TKM_WINDOW = (TIEPIE_HW_TK_INWINDOW or TIEPIE_HW_TK_OUTWINDOW or TIEPIE_HW_TK_ENTERWINDOW or TIEPIE_HW_TK_EXITWINDOW); // All window triggers
  TIEPIE_HW_TKM_PULSEWIDTH = (TIEPIE_HW_TK_PULSEWIDTHPOSITIVE or TIEPIE_HW_TK_PULSEWIDTHNEGATIVE or TIEPIE_HW_TK_PULSEWIDTHEITHER); // All pulse width triggers
  TIEPIE_HW_TKM_RUNTPULSE = (TIEPIE_HW_TK_RUNTPULSEPOSITIVE or TIEPIE_HW_TK_RUNTPULSENEGATIVE or TIEPIE_HW_TK_RUNTPULSEEITHER); // All runt pulse triggers
  TIEPIE_HW_TKM_PULSE = (TIEPIE_HW_TKM_PULSEWIDTH or TIEPIE_HW_TKM_RUNTPULSE); // All pulse triggers
  TIEPIE_HW_TKM_INTERVAL = (TIEPIE_HW_TK_INTERVALRISING or TIEPIE_HW_TK_INTERVALFALLING); // All interval triggers
  TIEPIE_HW_TKM_TIME = (TIEPIE_HW_TKM_PULSEWIDTH or TIEPIE_HW_TKM_WINDOW or TIEPIE_HW_TKM_RUNTPULSE or TIEPIE_HW_TKM_INTERVAL); // All trigger kinds that may have a time property.
  TIEPIE_HW_TKM_ALL = ((1 shl TIEPIE_HW_TKN_COUNT) - 1); // All trigger kinds

  TIEPIE_HW_TLMN_COUNT = 2; // Number of trigger level modes

  TIEPIE_HW_TLMB_RELATIVE = 0; // Bit number for Trigger Level Mode Relative
  TIEPIE_HW_TLMB_ABSOLUTE = 1; // Bit number for Trigger Level Mode Absolute

  TIEPIE_HW_TLM_UNKNOWN = 0; // Trigger Level Mode unknown/invalid
  TIEPIE_HW_TLM_RELATIVE = (1 shl TIEPIE_HW_TLMB_RELATIVE); // Trigger Level Mode Relative
  TIEPIE_HW_TLM_ABSOLUTE = (1 shl TIEPIE_HW_TLMB_ABSOLUTE); // Trigger Level Mode Absolute

  TIEPIE_HW_TLMM_NONE = 0; // None
  TIEPIE_HW_TLMM_ALL = ((1 shl TIEPIE_HW_TLMN_COUNT) - 1); // All Trigger Level Modes

  TIEPIE_HW_TO_INFINITY = (-1.0); // No time out, wait infinitely

  TIEPIE_HW_TOEN_COUNT = 6; // Number of trigger output events

  TIEPIE_HW_TOEB_GENERATOR_START = 0; // Bit number for Trigger Output Event Generator Start
  TIEPIE_HW_TOEB_GENERATOR_STOP = 1; // Bit number for Trigger Output Event Generator Stop
  TIEPIE_HW_TOEB_GENERATOR_NEWPERIOD = 2; // Bit number for Trigger Output Event Generator New Period
  TIEPIE_HW_TOEB_OSCILLOSCOPE_RUNNING = 3; // Bit number for Trigger Output Event Oscilloscope Running
  TIEPIE_HW_TOEB_OSCILLOSCOPE_TRIGGERED = 4; // Bit number for Trigger Output Event Oscilloscope Triggered
  TIEPIE_HW_TOEB_MANUAL = 5; // Bit number for Trigger Output Event Manual Trigger

  TIEPIE_HW_TOE_UNKNOWN = 0; // Trigger Output Event unknown
  TIEPIE_HW_TOE_GENERATOR_START = (1 shl TIEPIE_HW_TOEB_GENERATOR_START); // Trigger Output Event Generator Start
  TIEPIE_HW_TOE_GENERATOR_STOP = (1 shl TIEPIE_HW_TOEB_GENERATOR_STOP); // Trigger Output Event Generator Stop
  TIEPIE_HW_TOE_GENERATOR_NEWPERIOD = (1 shl TIEPIE_HW_TOEB_GENERATOR_NEWPERIOD); // Trigger Output Event Generator New Period
  TIEPIE_HW_TOE_OSCILLOSCOPE_RUNNING = (1 shl TIEPIE_HW_TOEB_OSCILLOSCOPE_RUNNING); // Trigger Output Event Oscilloscope Running
  TIEPIE_HW_TOE_OSCILLOSCOPE_TRIGGERED = (1 shl TIEPIE_HW_TOEB_OSCILLOSCOPE_TRIGGERED); // Trigger Output Event Oscilloscope Triggered
  TIEPIE_HW_TOE_MANUAL = (1 shl TIEPIE_HW_TOEB_MANUAL); // Trigger Output Event Manual Trigger

  TIEPIE_HW_TOEM_NONE = 0; // No trigger output events
  TIEPIE_HW_TOEM_GENERATOR = (TIEPIE_HW_TOE_GENERATOR_START or TIEPIE_HW_TOE_GENERATOR_STOP or TIEPIE_HW_TOE_GENERATOR_NEWPERIOD); // All generator trigger output events
  TIEPIE_HW_TOEM_OSCILLOSCOPE = (TIEPIE_HW_TOE_OSCILLOSCOPE_RUNNING or TIEPIE_HW_TOE_OSCILLOSCOPE_TRIGGERED); // All oscilloscope trigger output events
  TIEPIE_HW_TOEM_ALL = ((1 shl TIEPIE_HW_TOEN_COUNT) - 1); // All trigger output events

  {$if defined( LINUX )}
  TIEPIE_HW_FILENAME_DEFAULT = 'libtiepie-hw.so.1.5';
  {$elseif defined( MACOS )}
  TIEPIE_HW_FILENAME_DEFAULT = 'libtiepie-hw.dylib';
  {$else}
  TIEPIE_HW_FILENAME_DEFAULT = 'libtiepie-hw';
  {$ifend}

type
  Ttiepie_hw_status = (
    TIEPIE_HW_STATUS_VALUE_MODIFIED = 2, // One of the parameters of the last called function was within the valid range but not available. The closest valid value is set.
    TIEPIE_HW_STATUS_VALUE_CLIPPED = 1, // One of the parameters of the last called function was outside the valid range and clipped to the closest limit.
    TIEPIE_HW_STATUS_SUCCESS = 0, // The function executed successfully.
    TIEPIE_HW_STATUS_UNSUCCESSFUL = -1, // An error occurred during execution of the last called function.
    TIEPIE_HW_STATUS_NOT_SUPPORTED = -2, // The requested functionality is not supported by the hardware.
    TIEPIE_HW_STATUS_INVALID_HANDLE = -3, // The handle to the device is invalid.
    TIEPIE_HW_STATUS_INVALID_VALUE = -4, // The requested value is invalid.
    TIEPIE_HW_STATUS_INVALID_CHANNEL = -5, // The requested channel number is invalid.
    TIEPIE_HW_STATUS_INVALID_TRIGGER_SOURCE = -6, // The requested trigger source is invalid.
    TIEPIE_HW_STATUS_INVALID_DEVICE_TYPE = -7, // The device type is invalid.
    TIEPIE_HW_STATUS_INVALID_DEVICE_INDEX = -8, // The device index is invalid, must be < tiepie_hw_devicelist_get_count().
    TIEPIE_HW_STATUS_INVALID_PRODUCT_ID = -9, // There is no device with the requested product ID.
    TIEPIE_HW_STATUS_INVALID_DEVICE_SERIALNUMBER = -10, // There is no device with the requested serial number.
    TIEPIE_HW_STATUS_OBJECT_GONE = -11, // The object indicated by the handle is no longer available.
    TIEPIE_HW_STATUS_INTERNAL_ADDRESS = -12, // The requested I<sup>2</sup>C address is an internally used address in the device.
    TIEPIE_HW_STATUS_NOT_CONTROLLABLE = -13, // The generator is currently not controllable, see #tiepie_hw_generator_is_controllable.
    TIEPIE_HW_STATUS_BIT_ERROR = -14, // The requested I<sup>2</sup>C operation generated a bit error.
    TIEPIE_HW_STATUS_NO_ACKNOWLEDGE = -15, // The requested I<sup>2</sup>C operation generated "No acknowledge".
    TIEPIE_HW_STATUS_INVALID_CONTAINED_DEVICE_SERIALNUMBER = -16, // A device with the requested serial number is not available in the combined instrument, see #tiepie_hw_devicelistitem_get_contained_serial_numbers.
    TIEPIE_HW_STATUS_INVALID_INPUT = -17, // The requested trigger input is invalid.
    TIEPIE_HW_STATUS_INVALID_OUTPUT = -18, // The requested trigger output is invalid.
    TIEPIE_HW_STATUS_NOT_AVAILABLE = -20, // With the current settings, the requested functionality is not available.
    TIEPIE_HW_STATUS_INVALID_FIRMWARE = -21, // The currently used firmware is not supported.
    TIEPIE_HW_STATUS_INVALID_INDEX = -22, // The requested index is invalid.
    TIEPIE_HW_STATUS_INVALID_EEPROM = -23, // The instrument's EEPROM content is damaged, please contact TiePie engineering support.
    TIEPIE_HW_STATUS_INITIALIZATION_FAILED = -24, // The instrument's initialization failed, please contact TiePie engineering support.
    TIEPIE_HW_STATUS_LIBRARY_NOT_INITIALIZED = -25, // The library is not initialized, see tiepie_hw_init().
    TIEPIE_HW_STATUS_NO_TRIGGER_ENABLED = -26, // The current setup requires a trigger input to be enabled.
    TIEPIE_HW_STATUS_SYNCHRONIZATION_FAILED = -29, // Synchronization of the instruments has failed.
    TIEPIE_HW_STATUS_INVALID_HS56_COMBINED_DEVICE = -30, // At least one Handyscope HS6 (DIFF) / WiFiScope WS6 (DIFF) must be located at the beginning or end of the CMI daisy chain.
    TIEPIE_HW_STATUS_MEASUREMENT_RUNNING = -31, // A measurement is already running.
    TIEPIE_HW_STATUS_WIRELESSTRIGGERMODULENOTCONNECTED = -32, // A wireless trigger module is not connected.
    TIEPIE_HW_STATUS_INITIALIZATION_ERROR_10001 = -10001, // Initialization error 10001
    TIEPIE_HW_STATUS_INITIALIZATION_ERROR_10002 = -10002, // Initialization error 10002
    TIEPIE_HW_STATUS_INITIALIZATION_ERROR_10003 = -10003, // Initialization error 10003
    TIEPIE_HW_STATUS_INITIALIZATION_ERROR_10004 = -10004, // Initialization error 10004
    TIEPIE_HW_STATUS_INITIALIZATION_ERROR_10005 = -10005, // Initialization error 10005
    TIEPIE_HW_STATUS_INITIALIZATION_ERROR_10006 = -10006, // Initialization error 10006
    TIEPIE_HW_STATUS_INITIALIZATION_ERROR_10007 = -10007 // Initialization error 10007
  );

  Ttiepie_hw_productid = (
    TIEPIE_HW_PRODUCTID_NONE = 0, // Unknown/invalid ID
    TIEPIE_HW_PRODUCTID_COMBI = 2, // Combined instrument
    TIEPIE_HW_PRODUCTID_HS3 = 13, // Handyscope HS3
    TIEPIE_HW_PRODUCTID_HS4 = 15, // Handyscope HS4
    TIEPIE_HW_PRODUCTID_HP3 = 18, // Handyprobe HP3
    TIEPIE_HW_PRODUCTID_TP450 = 19, // TP450
    TIEPIE_HW_PRODUCTID_HS4D = 20, // Handyscope HS4 DIFF
    TIEPIE_HW_PRODUCTID_HS5 = 22, // Handyscope HS5
    TIEPIE_HW_PRODUCTID_HS6 = 24, // Handyscope HS6
    TIEPIE_HW_PRODUCTID_HS6D = 25, // Handyscope HS6 DIFF
    TIEPIE_HW_PRODUCTID_ATS610004D = 31, // ATS610004D
    TIEPIE_HW_PRODUCTID_ATS605004D = 32, // ATS605004D
    TIEPIE_HW_PRODUCTID_WS6 = 34, // WiFiScope WS6
    TIEPIE_HW_PRODUCTID_WS5 = 35, // WiFiScope WS5
    TIEPIE_HW_PRODUCTID_WS6D = 36, // WiFiScope WS6D
    TIEPIE_HW_PRODUCTID_ATS610004DW = 37, // ATS610004DW
    TIEPIE_HW_PRODUCTID_ATS605004DW = 38, // ATS605004DW
    TIEPIE_HW_PRODUCTID_WS4D = 39, // WiFiScope WS4D
    TIEPIE_HW_PRODUCTID_ATS5004DW = 40, // ATS5004DW
    TIEPIE_HW_PRODUCTID_ATS5004 = 42, // ATS5004
    TIEPIE_HW_PRODUCTID_ATS5004D = 43 // ATS5004D
  );

  Ttiepie_hw_event = (
    TIEPIE_HW_EVENT_INVALID = 0, // This event ID value should not occur.
    TIEPIE_HW_EVENT_OBJECT_REMOVED = 1, // Event ID for the event indicating that an object was removed.
    TIEPIE_HW_EVENT_OSCILLOSCOPE_DATA_READY = 2, // Event ID for the event indicating that the \ref tiepie_hw_oscilloscope_is_data_ready "oscilloscope measurement is ready".
    TIEPIE_HW_EVENT_OSCILLOSCOPE_DATA_OVERFLOW = 3, // Event ID for the event indicating that \ref tiepie_hw_oscilloscope_is_data_overflow "data overflow" occurred during a streaming measurement.
    TIEPIE_HW_EVENT_OSCILLOSCOPE_SURE_CONNECT_COMPLETED = 4, // Event ID for the event indicating that the \ref scp_ct "connection test is ready".
    TIEPIE_HW_EVENT_OSCILLOSCOPE_TRIGGERED = 5, // Event ID for the event indicating that the oscilloscope has triggered.
    TIEPIE_HW_EVENT_GENERATOR_BURST_COMPLETED = 6, // Event ID for the event indicating that the \ref gen_burst "generator burst is completed".
    TIEPIE_HW_EVENT_GENERATOR_CONTROLLABLE_CHANGED = 7, // Event ID for the event indicating that the \ref tiepie_hw_generator_is_controllable "generator controllable state" has changed.
    TIEPIE_HW_EVENT_SERVER_STATUS_CHANGED = 8, // Event ID for the event indicating that the \ref net_srv "server" state has changed.
    TIEPIE_HW_EVENT_OSCILLOSCOPE_SAFEGROUND_ERROR = 9, // Event ID for the event indicating that the oscilloscope channel \ref scp_ch_safeground "SafeGround" was disabled because of a too large ground current. The value parameter of the event contains the channel number (\c 0 to \c ChannelCount-1).
    TIEPIE_HW_EVENT_DEVICE_BATTERY_STATUS_CHANGED = 11, // Event ID for the event indicating that the \ref dev_battery "battery" state has changed.
    TIEPIE_HW_EVENT_OSCILLOSCOPE_WIRELESS_SYNCHRONIZATION_ERROR = 12 // Event ID for the event indicating that an error occurred with wireless synchronization of scopes
  );

  Ttiepie_hw_demosignal = (
    TIEPIE_HW_DEMOSIGNAL_NONE = 0, // Disable signal
    TIEPIE_HW_DEMOSIGNAL_SINE = 1, // Generate a sine wave signal.
    TIEPIE_HW_DEMOSIGNAL_TRIANGLE = 2, // Generate a triangular signal.
    TIEPIE_HW_DEMOSIGNAL_SQUARE = 3, // Generate a square wave signal.
    TIEPIE_HW_DEMOSIGNAL_DC = 4, // Generate a DC level.
    TIEPIE_HW_DEMOSIGNAL_RUNT = 5, // Generate a pulse signal with a runt pulse
    TIEPIE_HW_DEMOSIGNAL_I2C_SCL = 100, // Generate an I2C SCL signal
    TIEPIE_HW_DEMOSIGNAL_I2C_SDA = 101, // Generate an I2C SDA signal
    TIEPIE_HW_DEMOSIGNAL_UART_9600_8N1 = 200, // Generate a UART signal, 9600 baud, 8 bits, no parity, 1 stop bit
    TIEPIE_HW_DEMOSIGNAL_UART_9600_7E2 = 201, // Generate a UART signal, 9600 baud, 7 bits, even parity, 2 stop bits
    TIEPIE_HW_DEMOSIGNAL_RS232_9600_8N1 = 300, // Generate a RS232 signal, 9600 baud, 8 bits, no parity, 1 stop bit
    TIEPIE_HW_DEMOSIGNAL_RS232_9600_8O15 = 301, // Generate a RS232 signal, 9600 baud, 8 bits, odd parity, 1.5 stop bits
    TIEPIE_HW_DEMOSIGNAL_RS485_9600_8N1 = 400, // Generate a RS458 signal, 9600 baud, 8 bits, no parity, 1 stop bit
    TIEPIE_HW_DEMOSIGNAL_RS485_9600_8N1MD = 401, // Generate a RS458 signal, 9600 baud, 8 bits, no parity, 1 stop bit MultiDrop
    TIEPIE_HW_DEMOSIGNAL_CAN_250000 = 500, // Generate a differential CAN signal, 250 kbps
    TIEPIE_HW_DEMOSIGNAL_CAN_250000_HI = 501, // Generate a CAN-High signal, 250 kbps
    TIEPIE_HW_DEMOSIGNAL_CAN_250000_LO = 502, // Generate a CAN-Low signal, 250 kbps
    TIEPIE_HW_DEMOSIGNAL_CANFD_250000 = 600, // Generate a differential CANFD signal, 250 kbps
    TIEPIE_HW_DEMOSIGNAL_CANFD_250000_HI = 601, // Generate a CANFD-High signal, 250 kbps
    TIEPIE_HW_DEMOSIGNAL_CANFD_250000_LO = 602, // Generate a CANFD-Low signal, 250 kbps
    TIEPIE_HW_DEMOSIGNAL_DMX512 = 700, // Generate a DMX 512 signal
    TIEPIE_HW_DEMOSIGNAL_LIN_19200 = 800, // Generate a LIN bus signal, 19200 bps
    TIEPIE_HW_DEMOSIGNAL_EV_CHARGING = 900 // Generate a EV charging signal.
  );

  Ttiepie_hw_handle = LongWord; // Generic handle.
  Ttiepie_hw_tristate = Byte; // TriState value one byte wide. See: TIEPIE_HW_TRISTATE_
  Ttiepie_hw_pointerarray = PPointer; // Pointer array See: hlp_ptrar
  Ttiepie_hw_devicelist_callback = procedure(data : Pointer; device_types : LongWord; serial_number : LongWord); cdecl;
  Ttiepie_hw_handle_callback = procedure(data : Pointer; handle : Ttiepie_hw_handle); cdecl;
  Ttiepie_hw_event_callback = procedure(data : Pointer; event : Ttiepie_hw_event; value : LongWord); cdecl;

  Ttiepie_hw_date = record
    year : Word;
    month : Byte;
    day : Byte;

    class operator Explicit(value: Ttiepie_hw_date): TDateTime;
  end;

  Ttiepie_hw_demo_info = record
    product_id : Ttiepie_hw_productid;
    name : PAnsiChar;
    name_short : PAnsiChar;
  end;

  Ttiepie_hw_version = record
    major : Word;
    minor : Word;
    patch : Word;
    build : Word;
    extra : PAnsiChar;

    class operator Implicit(value: Ttiepie_hw_version): Boolean;
  end;

  Ptiepie_hw_version = ^Ttiepie_hw_version;
  Ptiepie_hw_demo_info = ^Ttiepie_hw_demo_info;
  Ptiepie_hw_handle = ^Ttiepie_hw_handle;
  PByteBool = ^ByteBool;
  PPSingle = ^PSingle;
  Ptiepie_hw_tristate = ^Ttiepie_hw_tristate;
  PUInt64 = ^UInt64;


{$ifdef LIBTIEPIE_HW_DYNAMIC}
type
  Ttiepie_hw_init = procedure; cdecl;
  Ttiepie_hw_is_initialized = function : ByteBool; cdecl;
  Ttiepie_hw_fini = procedure; cdecl;
  Ttiepie_hw_get_version = function : Ptiepie_hw_version; cdecl;
  Ttiepie_hw_get_config = function(buffer : PByte; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_get_last_status = function : Ttiepie_hw_status; cdecl;
  Ttiepie_hw_get_last_status_str = function : PAnsiChar; cdecl;
  Ttiepie_hw_devicelist_update = procedure; cdecl;
  Ttiepie_hw_devicelist_get_usb_hotplug_detect_enabled = function : ByteBool; cdecl;
  Ttiepie_hw_devicelist_set_usb_hotplug_detect_enabled = function(value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_devicelist_get_count = function : LongWord; cdecl;
  Ttiepie_hw_devicelist_get_item_by_product_id = function(product_id : Ttiepie_hw_productid) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_devicelist_get_item_by_index = function(index : LongWord) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_devicelist_get_item_by_serial_number = function(serial_number : LongWord) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_devicelistitem_open_device = function(handle : Ttiepie_hw_handle; device_type : LongWord) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_devicelistitem_open_oscilloscope = function(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_devicelistitem_open_generator = function(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_devicelist_get_demo_device_info = function : Ptiepie_hw_demo_info; cdecl;
  Ttiepie_hw_devicelist_create_demo_device = function(product_id : Ttiepie_hw_productid) : LongWord; cdecl;
  Ttiepie_hw_devicelist_create_combined_device = function(handles : Ptiepie_hw_handle; count : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelist_create_and_open_combined_device = function(handles : Ptiepie_hw_handle; count : LongWord) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_devicelist_remove_device = procedure(serial_number : LongWord; force : ByteBool); cdecl;
  Ttiepie_hw_devicelist_remove_unused_devices = procedure; cdecl;
  Ttiepie_hw_devicelistitem_is_demo = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_devicelistitem_can_open = function(handle : Ttiepie_hw_handle; device_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_devicelistitem_opened_by = function(handle : Ttiepie_hw_handle; device_type : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitem_get_product_id = function(handle : Ttiepie_hw_handle) : Ttiepie_hw_productid; cdecl;
  Ttiepie_hw_devicelistitem_get_name = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitem_get_name_short = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitem_get_name_shortest = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitem_get_calibration_date = function(handle : Ttiepie_hw_handle) : Ttiepie_hw_date; cdecl;
  Ttiepie_hw_devicelistitem_get_serial_number = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_devicelistitem_get_ip_address = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitem_get_ip_port = function(handle : Ttiepie_hw_handle) : Word; cdecl;
  Ttiepie_hw_devicelistitem_has_server = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_devicelistitem_get_server = function(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_devicelistitem_get_types = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_devicelistitem_get_contained_serial_numbers = function(handle : Ttiepie_hw_handle; buffer : PLongWord; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitemcombined_get_product_id = function(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Ttiepie_hw_productid; cdecl;
  Ttiepie_hw_devicelistitemcombined_get_name = function(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitemcombined_get_name_short = function(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitemcombined_get_name_shortest = function(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_devicelistitemcombined_get_calibration_date = function(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Ttiepie_hw_date; cdecl;
  Ttiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count = function(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Word; cdecl;
  Ttiepie_hw_devicelist_set_callback_device_added = procedure(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl;
  Ttiepie_hw_devicelist_set_callback_device_removed = procedure(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl;
  Ttiepie_hw_devicelist_set_callback_device_can_open_changed = procedure(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl;
  Ttiepie_hw_network_get_auto_detect_enabled = function : ByteBool; cdecl;
  Ttiepie_hw_network_set_auto_detect_enabled = function(value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_network_servers_add = function(url : PAnsiChar; length : LongWord; handle : Ptiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_network_servers_remove = function(url : PAnsiChar; length : LongWord; force : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_network_servers_get_count = function : LongWord; cdecl;
  Ttiepie_hw_network_servers_get_by_index = function(index : LongWord) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_network_servers_get_by_url = function(url : PAnsiChar; length : LongWord) : Ttiepie_hw_handle; cdecl;
  Ttiepie_hw_network_servers_set_callback_added = procedure(callback : Ttiepie_hw_handle_callback; data : Pointer); cdecl;
  Ttiepie_hw_object_close = procedure(handle : Ttiepie_hw_handle); cdecl;
  Ttiepie_hw_object_is_removed = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_object_get_interfaces = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_object_set_event_callback = procedure(handle : Ttiepie_hw_handle; callback : Ttiepie_hw_event_callback; data : Pointer); cdecl;
  Ttiepie_hw_device_get_calibration_date = function(handle : Ttiepie_hw_handle) : Ttiepie_hw_date; cdecl;
  Ttiepie_hw_device_get_serial_number = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_device_get_ip_address = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_device_get_ip_port = function(handle : Ttiepie_hw_handle) : Word; cdecl;
  Ttiepie_hw_device_get_product_id = function(handle : Ttiepie_hw_handle) : Ttiepie_hw_productid; cdecl;
  Ttiepie_hw_device_get_type = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_device_get_name = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_device_get_name_short = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_device_get_name_shortest = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_device_has_wireless_trigger_module = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_device_has_battery = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_device_has_battery_ex = function(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_device_get_battery_charge = function(handle : Ttiepie_hw_handle) : ShortInt; cdecl;
  Ttiepie_hw_device_get_battery_charge_ex = function(handle : Ttiepie_hw_handle; serial_number : LongWord) : ShortInt; cdecl;
  Ttiepie_hw_device_get_battery_time_to_empty = function(handle : Ttiepie_hw_handle) : LongInt; cdecl;
  Ttiepie_hw_device_get_battery_time_to_empty_ex = function(handle : Ttiepie_hw_handle; serial_number : LongWord) : LongInt; cdecl;
  Ttiepie_hw_device_get_battery_time_to_full = function(handle : Ttiepie_hw_handle) : LongInt; cdecl;
  Ttiepie_hw_device_get_battery_time_to_full_ex = function(handle : Ttiepie_hw_handle; serial_number : LongWord) : LongInt; cdecl;
  Ttiepie_hw_device_is_battery_charger_connected = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_device_is_battery_charger_connected_ex = function(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_device_is_battery_charging = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_device_is_battery_charging_ex = function(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_device_is_battery_broken = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_device_is_battery_broken_ex = function(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_device_trigger_get_input_count = function(handle : Ttiepie_hw_handle) : Word; cdecl;
  Ttiepie_hw_device_trigger_get_input_index_by_id = function(handle : Ttiepie_hw_handle; id : LongWord) : Word; cdecl;
  Ttiepie_hw_oscilloscope_trigger_input_is_triggered = function(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl;
  Ttiepie_hw_device_trigger_input_get_enabled = function(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl;
  Ttiepie_hw_device_trigger_input_set_enabled = function(handle : Ttiepie_hw_handle; input : Word; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_device_trigger_input_get_kinds = function(handle : Ttiepie_hw_handle; input : Word) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_trigger_input_get_kinds_ex = function(handle : Ttiepie_hw_handle; input : Word; measure_mode : LongWord) : UInt64; cdecl;
  Ttiepie_hw_device_trigger_input_get_kind = function(handle : Ttiepie_hw_handle; input : Word) : UInt64; cdecl;
  Ttiepie_hw_device_trigger_input_set_kind = function(handle : Ttiepie_hw_handle; input : Word; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_device_trigger_input_is_available = function(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_trigger_input_is_available_ex = function(handle : Ttiepie_hw_handle; input : Word; measure_mode : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_device_trigger_input_get_id = function(handle : Ttiepie_hw_handle; input : Word) : LongWord; cdecl;
  Ttiepie_hw_device_trigger_input_get_name = function(handle : Ttiepie_hw_handle; input : Word; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_device_trigger_get_output_count = function(handle : Ttiepie_hw_handle) : Word; cdecl;
  Ttiepie_hw_device_trigger_get_output_index_by_id = function(handle : Ttiepie_hw_handle; id : LongWord) : Word; cdecl;
  Ttiepie_hw_device_trigger_output_get_enabled = function(handle : Ttiepie_hw_handle; output : Word) : ByteBool; cdecl;
  Ttiepie_hw_device_trigger_output_set_enabled = function(handle : Ttiepie_hw_handle; output : Word; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_device_trigger_output_get_events = function(handle : Ttiepie_hw_handle; output : Word) : UInt64; cdecl;
  Ttiepie_hw_device_trigger_output_get_event = function(handle : Ttiepie_hw_handle; output : Word) : UInt64; cdecl;
  Ttiepie_hw_device_trigger_output_set_event = function(handle : Ttiepie_hw_handle; output : Word; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_device_trigger_output_get_id = function(handle : Ttiepie_hw_handle; output : Word) : LongWord; cdecl;
  Ttiepie_hw_device_trigger_output_get_name = function(handle : Ttiepie_hw_handle; output : Word; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_device_trigger_output_trigger = function(handle : Ttiepie_hw_handle; output : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_get_channel_count = function(handle : Ttiepie_hw_handle) : Word; cdecl;
  Ttiepie_hw_oscilloscope_channel_is_available = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_is_available_ex = function(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; sample_rate : Double; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_connector_type = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_is_differential = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_is_isolated = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_impedance = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_bandwidths = function(handle : Ttiepie_hw_handle; ch : Word; list : PDouble; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_bandwidth = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_set_bandwidth = function(handle : Ttiepie_hw_handle; ch : Word; bandwidth : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_couplings = function(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_coupling = function(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_set_coupling = function(handle : Ttiepie_hw_handle; ch : Word; coupling : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_enabled = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_set_enabled = function(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_auto_ranging = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_set_auto_ranging = function(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_ranges = function(handle : Ttiepie_hw_handle; ch : Word; list : PDouble; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_ranges_ex = function(handle : Ttiepie_hw_handle; ch : Word; coupling : UInt64; list : PDouble; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_range = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_set_range = function(handle : Ttiepie_hw_handle; ch : Word; range : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_has_safeground = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_safeground_enabled = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_set_safeground_enabled = function(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_safeground_threshold_min = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_safeground_threshold_max = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_safeground_threshold = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_set_safeground_threshold = function(handle : Ttiepie_hw_handle; ch : Word; threshold : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_verify_safeground_threshold = function(handle : Ttiepie_hw_handle; ch : Word; threshold : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_has_trigger = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_has_trigger_ex = function(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_is_available = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_is_available_ex = function(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; sample_rate : Double; resolution : Byte; channel_enabled : PByteBool; channel_trigger_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_is_triggered = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_enabled = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_set_enabled = function(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_kinds = function(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_kinds_ex = function(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_kind = function(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_set_kind = function(handle : Ttiepie_hw_handle; ch : Word; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_level_modes = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_level_mode = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_set_level_mode = function(handle : Ttiepie_hw_handle; ch : Word; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_level_count = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_level = function(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_set_level = function(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_hysteresis = function(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_set_hysteresis = function(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_conditions = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_conditions_ex = function(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; trigger_kind : UInt64) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_condition = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_set_condition = function(handle : Ttiepie_hw_handle; ch : Word; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_time_count = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_get_time = function(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_set_time = function(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_verify_time = function(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_trigger_verify_time_ex = function(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double; measure_mode : LongWord; sample_rate : Double; trigger_kind : UInt64; trigger_condition : LongWord) : Double; cdecl;
  Ttiepie_hw_oscilloscope_is_demo = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_get_signal = function(handle : Ttiepie_hw_handle; ch : Word) : Ttiepie_hw_demosignal; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_set_signal = function(handle : Ttiepie_hw_handle; ch : Word; value : Ttiepie_hw_demosignal) : Ttiepie_hw_demosignal; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_has_amplitude = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_get_amplitude = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_set_amplitude = function(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_has_frequency = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_get_frequency = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_set_frequency = function(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_has_offset = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_get_offset = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_set_offset = function(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_has_symmetry = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_get_symmetry = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_set_symmetry = function(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_has_phase = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_get_phase = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_set_phase = function(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_has_noise_enabled = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_get_noise_enabled = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_demo_set_noise_enabled = function(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_get_data = function(handle : Ttiepie_hw_handle; buffers : PPSingle; channel_count : Word; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_1ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_2ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; buffer_ch2 : PSingle; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_3ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; buffer_ch2 : PSingle; buffer_ch3 : PSingle; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_4ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; buffer_ch2 : PSingle; buffer_ch3 : PSingle; buffer_ch4 : PSingle; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_5ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; buffer_ch2 : PSingle; buffer_ch3 : PSingle; buffer_ch4 : PSingle; buffer_ch5 : PSingle; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_6ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; buffer_ch2 : PSingle; buffer_ch3 : PSingle; buffer_ch4 : PSingle; buffer_ch5 : PSingle; buffer_ch6 : PSingle; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_7ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; buffer_ch2 : PSingle; buffer_ch3 : PSingle; buffer_ch4 : PSingle; buffer_ch5 : PSingle; buffer_ch6 : PSingle; buffer_ch7 : PSingle; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_8ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; buffer_ch2 : PSingle; buffer_ch3 : PSingle; buffer_ch4 : PSingle; buffer_ch5 : PSingle; buffer_ch6 : PSingle; buffer_ch7 : PSingle; buffer_ch8 : PSingle; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_valid_pre_sample_count = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_data_value_range = procedure(handle : Ttiepie_hw_handle; ch : Word; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_oscilloscope_channel_get_data_value_min = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_data_value_max = function(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw = function(handle : Ttiepie_hw_handle; buffers : PPointer; channel_count : Word; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw_1ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw_2ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; buffer_ch2 : Pointer; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw_3ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; buffer_ch2 : Pointer; buffer_ch3 : Pointer; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw_4ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; buffer_ch2 : Pointer; buffer_ch3 : Pointer; buffer_ch4 : Pointer; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw_5ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; buffer_ch2 : Pointer; buffer_ch3 : Pointer; buffer_ch4 : Pointer; buffer_ch5 : Pointer; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw_6ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; buffer_ch2 : Pointer; buffer_ch3 : Pointer; buffer_ch4 : Pointer; buffer_ch5 : Pointer; buffer_ch6 : Pointer; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw_7ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; buffer_ch2 : Pointer; buffer_ch3 : Pointer; buffer_ch4 : Pointer; buffer_ch5 : Pointer; buffer_ch6 : Pointer; buffer_ch7 : Pointer; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_data_raw_8ch = function(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; buffer_ch2 : Pointer; buffer_ch3 : Pointer; buffer_ch4 : Pointer; buffer_ch5 : Pointer; buffer_ch6 : Pointer; buffer_ch7 : Pointer; buffer_ch8 : Pointer; start_index : UInt64; sample_count : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_data_raw_type = function(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_data_raw_value_range = procedure(handle : Ttiepie_hw_handle; ch : Word; min : PInt64; zero : PInt64; max : PInt64); cdecl;
  Ttiepie_hw_oscilloscope_channel_get_data_raw_value_min = function(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_data_raw_value_zero = function(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl;
  Ttiepie_hw_oscilloscope_channel_get_data_raw_value_max = function(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl;
  Ttiepie_hw_oscilloscope_channel_is_range_max_reachable = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_start = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_stop = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_force_trigger = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_get_measure_modes = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_measure_mode = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_set_measure_mode = function(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_is_running = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_is_triggered = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_is_timeout_triggered = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_is_force_triggered = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_is_data_ready = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_is_data_overflow = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_get_auto_resolution_modes = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_auto_resolution_mode = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_set_auto_resolution_mode = function(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_resolutions = function(handle : Ttiepie_hw_handle; list : PByte; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_resolution = function(handle : Ttiepie_hw_handle) : Byte; cdecl;
  Ttiepie_hw_oscilloscope_set_resolution = function(handle : Ttiepie_hw_handle; value : Byte) : Byte; cdecl;
  Ttiepie_hw_oscilloscope_is_resolution_enhanced = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_is_resolution_enhanced_ex = function(handle : Ttiepie_hw_handle; value : Byte) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_sources = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_source = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_set_clock_source = function(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_source_frequencies = function(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_source_frequencies_ex = function(handle : Ttiepie_hw_handle; value : LongWord; list : PDouble; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_source_frequency = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_oscilloscope_set_clock_source_frequency = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_outputs = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_output = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_set_clock_output = function(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_output_frequencies = function(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_output_frequencies_ex = function(handle : Ttiepie_hw_handle; clock_output : LongWord; list : PDouble; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_clock_output_frequency = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_oscilloscope_set_clock_output_frequency = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_get_sample_rate_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_oscilloscope_get_sample_rate = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_oscilloscope_set_sample_rate = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_verify_sample_rate = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_verify_sample_rate_ex = function(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : Double; cdecl;
  Ttiepie_hw_oscilloscope_verify_sample_rates_ex = procedure(handle : Ttiepie_hw_handle; values : PDouble; count : LongWord; measure_mode : LongWord; auto_resolution_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word); cdecl;
  Ttiepie_hw_oscilloscope_get_record_length_max = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_record_length_max_ex = function(handle : Ttiepie_hw_handle; measure_mode : LongWord; resolution : Byte) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_record_length = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_set_record_length = function(handle : Ttiepie_hw_handle; record_length : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_verify_record_length = function(handle : Ttiepie_hw_handle; record_length : UInt64) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_verify_record_length_ex = function(handle : Ttiepie_hw_handle; record_length : UInt64; measure_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : UInt64; cdecl;
  Ttiepie_hw_oscilloscope_get_pre_sample_ratio = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_oscilloscope_set_pre_sample_ratio = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_get_segment_count_max = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_segment_count_max_ex = function(handle : Ttiepie_hw_handle; measure_mode : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_get_segment_count = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_set_segment_count = function(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_verify_segment_count = function(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_verify_segment_count_ex = function(handle : Ttiepie_hw_handle; value : LongWord; measure_mode : LongWord; record_length : UInt64; channel_enabled : PByteBool; channel_count : Word) : LongWord; cdecl;
  Ttiepie_hw_oscilloscope_has_trigger = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_has_trigger_ex = function(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_trigger_get_timeout = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_set_timeout = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_verify_timeout = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_verify_timeout_ex = function(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; sample_rate : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_has_delay = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_trigger_has_delay_ex = function(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_trigger_get_delay_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_get_delay_max_ex = function(handle : Ttiepie_hw_handle; measure_mode : LongWord; sample_rate : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_get_delay = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_set_delay = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_verify_delay = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_trigger_verify_delay_ex = function(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; sample_rate : Double) : Double; cdecl;
  Ttiepie_hw_oscilloscope_has_presamples_valid = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_has_presamples_valid_ex = function(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_get_presamples_valid = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_set_presamples_valid = function(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_has_sureconnect = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_channel_has_sureconnect = function(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_start_sureconnect = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_start_sureconnect_ex = function(handle : Ttiepie_hw_handle; channel_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_is_sureconnect_completed = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_oscilloscope_get_sureconnect_data = function(handle : Ttiepie_hw_handle; buffer : Ptiepie_hw_tristate; channel_count : Word) : Word; cdecl;
  Ttiepie_hw_generator_get_connector_type = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_generator_is_differential = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_impedance = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_resolution = function(handle : Ttiepie_hw_handle) : Byte; cdecl;
  Ttiepie_hw_generator_get_output_value_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_output_value_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_output_value_min_max = procedure(handle : Ttiepie_hw_handle; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_is_controllable = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_is_running = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_status = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_generator_get_output_enable = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_set_output_enable = function(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_output_invert = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_output_invert = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_set_output_invert = function(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_generator_start = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_stop = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_signal_types = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_generator_get_signal_type = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_generator_set_signal_type = function(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_generator_has_amplitude = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_amplitude_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_amplitude_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_amplitude_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_amplitude_min_max_ex = procedure(handle : Ttiepie_hw_handle; signal_type : LongWord; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_amplitude = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_amplitude = function(handle : Ttiepie_hw_handle; amplitude : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_amplitude = function(handle : Ttiepie_hw_handle; amplitude : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_amplitude_ex = function(handle : Ttiepie_hw_handle; amplitude : Double; signal_type : LongWord; amplitude_range_index : LongWord; offset : Double; output_invert : ByteBool) : Double; cdecl;
  Ttiepie_hw_generator_get_amplitude_ranges = function(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_generator_get_amplitude_range = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_amplitude_range = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_get_amplitude_auto_ranging = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_set_amplitude_auto_ranging = function(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_offset = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_offset_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_offset_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_offset_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_offset_min_max_ex = procedure(handle : Ttiepie_hw_handle; signal_type : LongWord; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_offset = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_offset = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_offset = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_offset_ex = function(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord; amplitude : Double; output_invert : ByteBool) : Double; cdecl;
  Ttiepie_hw_generator_get_frequency_modes = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_generator_get_frequency_modes_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord) : LongWord; cdecl;
  Ttiepie_hw_generator_get_frequency_mode = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_generator_set_frequency_mode = function(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
  Ttiepie_hw_generator_has_frequency = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_frequency_ex = function(handle : Ttiepie_hw_handle; frequency_mode : LongWord; signal_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_frequency_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_frequency_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_frequency_min_max = procedure(handle : Ttiepie_hw_handle; frequency_mode : LongWord; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_frequency_min_max_ex = procedure(handle : Ttiepie_hw_handle; frequency_mode : LongWord; signal_type : LongWord; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_frequency = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_frequency = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_frequency = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_frequency_ex = function(handle : Ttiepie_hw_handle; value : Double; frequency_mode : LongWord; signal_type : LongWord; data_length : UInt64; width : Double) : Double; cdecl;
  Ttiepie_hw_generator_has_phase = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_phase_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_phase_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_phase_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_phase_min_max_ex = procedure(handle : Ttiepie_hw_handle; signal_type : LongWord; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_phase = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_phase = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_phase = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_phase_ex = function(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord) : Double; cdecl;
  Ttiepie_hw_generator_has_symmetry = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_symmetry_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_symmetry_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_symmetry_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_symmetry_min_max_ex = procedure(handle : Ttiepie_hw_handle; signal_type : LongWord; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_symmetry = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_symmetry = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_symmetry = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_symmetry_ex = function(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord) : Double; cdecl;
  Ttiepie_hw_generator_has_width = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_width_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_width_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_width_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_width_min_max_ex = procedure(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency : Double; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_width = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_width = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_width = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_width_ex = function(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord; signal_frequency : Double) : Double; cdecl;
  Ttiepie_hw_generator_has_edge_time = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_edge_time_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_leading_edge_time_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_leading_edge_time_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_leading_edge_time_min_max_ex = procedure(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency : Double; symmetry : Double; width : Double; trailing_edge_time : Double; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_leading_edge_time = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_leading_edge_time = function(handle : Ttiepie_hw_handle; leading_edge_time : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_leading_edge_time = function(handle : Ttiepie_hw_handle; leading_edge_time : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_leading_edge_time_ex = function(handle : Ttiepie_hw_handle; leading_edge_time : Double; signal_type : LongWord; signal_frequency : Double; symmetry : Double; width : Double; trailing_edge_time : Double) : Double; cdecl;
  Ttiepie_hw_generator_get_trailing_edge_time_min = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_trailing_edge_time_max = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_get_trailing_edge_time_min_max_ex = procedure(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency : Double; symmetry : Double; width : Double; leading_edge_time : Double; min : PDouble; max : PDouble); cdecl;
  Ttiepie_hw_generator_get_trailing_edge_time = function(handle : Ttiepie_hw_handle) : Double; cdecl;
  Ttiepie_hw_generator_set_trailing_edge_time = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_trailing_edge_time = function(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
  Ttiepie_hw_generator_verify_trailing_edge_time_ex = function(handle : Ttiepie_hw_handle; trailing_edge_time : Double; signal_type : LongWord; signal_frequency : Double; symmetry : Double; width : Double; leading_edge_time : Double) : Double; cdecl;
  Ttiepie_hw_generator_has_data = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_has_data_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_data_length_min = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_data_length_max = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_data_length_min_max_ex = procedure(handle : Ttiepie_hw_handle; signal_type : LongWord; min : PUInt64; max : PUInt64); cdecl;
  Ttiepie_hw_generator_get_data_length = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_verify_data_length = function(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_generator_verify_data_length_ex = function(handle : Ttiepie_hw_handle; value : UInt64; signal_type : LongWord) : UInt64; cdecl;
  Ttiepie_hw_generator_set_data = procedure(handle : Ttiepie_hw_handle; buffer : PSingle; sample_count : UInt64); cdecl;
  Ttiepie_hw_generator_set_data_ex = procedure(handle : Ttiepie_hw_handle; buffer : PSingle; sample_count : UInt64; signal_type : LongWord); cdecl;
  Ttiepie_hw_generator_get_data_raw_type = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_generator_get_data_raw_value_range = procedure(handle : Ttiepie_hw_handle; min : PInt64; zero : PInt64; max : PInt64); cdecl;
  Ttiepie_hw_generator_get_data_raw_value_min = function(handle : Ttiepie_hw_handle) : Int64; cdecl;
  Ttiepie_hw_generator_get_data_raw_value_zero = function(handle : Ttiepie_hw_handle) : Int64; cdecl;
  Ttiepie_hw_generator_get_data_raw_value_max = function(handle : Ttiepie_hw_handle) : Int64; cdecl;
  Ttiepie_hw_generator_set_data_raw = procedure(handle : Ttiepie_hw_handle; buffer : Pointer; sample_count : UInt64); cdecl;
  Ttiepie_hw_generator_set_data_raw_ex = procedure(handle : Ttiepie_hw_handle; buffer : Pointer; sample_count : UInt64; signal_type : LongWord); cdecl;
  Ttiepie_hw_generator_get_modes = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_modes_ex = function(handle : Ttiepie_hw_handle; signal_type : LongWord; frequency_mode : LongWord) : UInt64; cdecl;
  Ttiepie_hw_generator_get_modes_native = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_mode = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_set_mode = function(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_generator_is_burst_active = function(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
  Ttiepie_hw_generator_get_burst_count_min = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_burst_count_max = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_burst_count_min_max_ex = procedure(handle : Ttiepie_hw_handle; generator_mode : UInt64; min : PUInt64; max : PUInt64); cdecl;
  Ttiepie_hw_generator_get_burst_count = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_set_burst_count = function(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_generator_get_burst_sample_count_min = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_burst_sample_count_max = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_burst_sample_count_min_max_ex = procedure(handle : Ttiepie_hw_handle; generator_mode : UInt64; min : PUInt64; max : PUInt64); cdecl;
  Ttiepie_hw_generator_get_burst_sample_count = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_set_burst_sample_count = function(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_generator_get_burst_segment_count_min = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_burst_segment_count_max = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_get_burst_segment_count_min_max_ex = procedure(handle : Ttiepie_hw_handle; generator_mode : UInt64; signal_type : LongWord; frequency_mode : LongWord; frequency : Double; data_length : UInt64; min : PUInt64; max : PUInt64); cdecl;
  Ttiepie_hw_generator_get_burst_segment_count = function(handle : Ttiepie_hw_handle) : UInt64; cdecl;
  Ttiepie_hw_generator_set_burst_segment_count = function(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_generator_verify_burst_segment_count = function(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
  Ttiepie_hw_generator_verify_burst_segment_count_ex = function(handle : Ttiepie_hw_handle; value : UInt64; generator_mode : UInt64; signal_type : LongWord; frequency_mode : LongWord; frequency : Double; data_length : UInt64) : UInt64; cdecl;
  Ttiepie_hw_server_connect = function(handle : Ttiepie_hw_handle; asynchronous : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_server_disconnect = function(handle : Ttiepie_hw_handle; force : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_server_remove = function(handle : Ttiepie_hw_handle; force : ByteBool) : ByteBool; cdecl;
  Ttiepie_hw_server_get_status = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_server_get_last_error = function(handle : Ttiepie_hw_handle) : LongWord; cdecl;
  Ttiepie_hw_server_get_url = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_server_get_id = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_server_get_ip_address = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_server_get_ip_port = function(handle : Ttiepie_hw_handle) : Word; cdecl;
  Ttiepie_hw_server_get_name = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_server_get_description = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_server_get_version = function(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
  Ttiepie_hw_pointerarray_new = function(length : LongWord) : Ttiepie_hw_pointerarray; cdecl;
  Ttiepie_hw_pointerarray_set = procedure(ptr : Ttiepie_hw_pointerarray; index : LongWord; value : Pointer); cdecl;
  Ttiepie_hw_pointerarray_delete = procedure(ptr : Ttiepie_hw_pointerarray); cdecl;

var
  tiepie_hw_init : Ttiepie_hw_init = nil;
  tiepie_hw_is_initialized : Ttiepie_hw_is_initialized = nil;
  tiepie_hw_fini : Ttiepie_hw_fini = nil;
  tiepie_hw_get_version : Ttiepie_hw_get_version = nil;
  tiepie_hw_get_config : Ttiepie_hw_get_config = nil;
  tiepie_hw_get_last_status : Ttiepie_hw_get_last_status = nil;
  tiepie_hw_get_last_status_str : Ttiepie_hw_get_last_status_str = nil;
  tiepie_hw_devicelist_update : Ttiepie_hw_devicelist_update = nil;
  tiepie_hw_devicelist_get_usb_hotplug_detect_enabled : Ttiepie_hw_devicelist_get_usb_hotplug_detect_enabled = nil;
  tiepie_hw_devicelist_set_usb_hotplug_detect_enabled : Ttiepie_hw_devicelist_set_usb_hotplug_detect_enabled = nil;
  tiepie_hw_devicelist_get_count : Ttiepie_hw_devicelist_get_count = nil;
  tiepie_hw_devicelist_get_item_by_product_id : Ttiepie_hw_devicelist_get_item_by_product_id = nil;
  tiepie_hw_devicelist_get_item_by_index : Ttiepie_hw_devicelist_get_item_by_index = nil;
  tiepie_hw_devicelist_get_item_by_serial_number : Ttiepie_hw_devicelist_get_item_by_serial_number = nil;
  tiepie_hw_devicelistitem_open_device : Ttiepie_hw_devicelistitem_open_device = nil;
  tiepie_hw_devicelistitem_open_oscilloscope : Ttiepie_hw_devicelistitem_open_oscilloscope = nil;
  tiepie_hw_devicelistitem_open_generator : Ttiepie_hw_devicelistitem_open_generator = nil;
  tiepie_hw_devicelist_get_demo_device_info : Ttiepie_hw_devicelist_get_demo_device_info = nil;
  tiepie_hw_devicelist_create_demo_device : Ttiepie_hw_devicelist_create_demo_device = nil;
  tiepie_hw_devicelist_create_combined_device : Ttiepie_hw_devicelist_create_combined_device = nil;
  tiepie_hw_devicelist_create_and_open_combined_device : Ttiepie_hw_devicelist_create_and_open_combined_device = nil;
  tiepie_hw_devicelist_remove_device : Ttiepie_hw_devicelist_remove_device = nil;
  tiepie_hw_devicelist_remove_unused_devices : Ttiepie_hw_devicelist_remove_unused_devices = nil;
  tiepie_hw_devicelistitem_is_demo : Ttiepie_hw_devicelistitem_is_demo = nil;
  tiepie_hw_devicelistitem_can_open : Ttiepie_hw_devicelistitem_can_open = nil;
  tiepie_hw_devicelistitem_opened_by : Ttiepie_hw_devicelistitem_opened_by = nil;
  tiepie_hw_devicelistitem_get_product_id : Ttiepie_hw_devicelistitem_get_product_id = nil;
  tiepie_hw_devicelistitem_get_name : Ttiepie_hw_devicelistitem_get_name = nil;
  tiepie_hw_devicelistitem_get_name_short : Ttiepie_hw_devicelistitem_get_name_short = nil;
  tiepie_hw_devicelistitem_get_name_shortest : Ttiepie_hw_devicelistitem_get_name_shortest = nil;
  tiepie_hw_devicelistitem_get_calibration_date : Ttiepie_hw_devicelistitem_get_calibration_date = nil;
  tiepie_hw_devicelistitem_get_serial_number : Ttiepie_hw_devicelistitem_get_serial_number = nil;
  tiepie_hw_devicelistitem_get_ip_address : Ttiepie_hw_devicelistitem_get_ip_address = nil;
  tiepie_hw_devicelistitem_get_ip_port : Ttiepie_hw_devicelistitem_get_ip_port = nil;
  tiepie_hw_devicelistitem_has_server : Ttiepie_hw_devicelistitem_has_server = nil;
  tiepie_hw_devicelistitem_get_server : Ttiepie_hw_devicelistitem_get_server = nil;
  tiepie_hw_devicelistitem_get_types : Ttiepie_hw_devicelistitem_get_types = nil;
  tiepie_hw_devicelistitem_get_contained_serial_numbers : Ttiepie_hw_devicelistitem_get_contained_serial_numbers = nil;
  tiepie_hw_devicelistitemcombined_get_product_id : Ttiepie_hw_devicelistitemcombined_get_product_id = nil;
  tiepie_hw_devicelistitemcombined_get_name : Ttiepie_hw_devicelistitemcombined_get_name = nil;
  tiepie_hw_devicelistitemcombined_get_name_short : Ttiepie_hw_devicelistitemcombined_get_name_short = nil;
  tiepie_hw_devicelistitemcombined_get_name_shortest : Ttiepie_hw_devicelistitemcombined_get_name_shortest = nil;
  tiepie_hw_devicelistitemcombined_get_calibration_date : Ttiepie_hw_devicelistitemcombined_get_calibration_date = nil;
  tiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count : Ttiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count = nil;
  tiepie_hw_devicelist_set_callback_device_added : Ttiepie_hw_devicelist_set_callback_device_added = nil;
  tiepie_hw_devicelist_set_callback_device_removed : Ttiepie_hw_devicelist_set_callback_device_removed = nil;
  tiepie_hw_devicelist_set_callback_device_can_open_changed : Ttiepie_hw_devicelist_set_callback_device_can_open_changed = nil;
  tiepie_hw_network_get_auto_detect_enabled : Ttiepie_hw_network_get_auto_detect_enabled = nil;
  tiepie_hw_network_set_auto_detect_enabled : Ttiepie_hw_network_set_auto_detect_enabled = nil;
  tiepie_hw_network_servers_add : Ttiepie_hw_network_servers_add = nil;
  tiepie_hw_network_servers_remove : Ttiepie_hw_network_servers_remove = nil;
  tiepie_hw_network_servers_get_count : Ttiepie_hw_network_servers_get_count = nil;
  tiepie_hw_network_servers_get_by_index : Ttiepie_hw_network_servers_get_by_index = nil;
  tiepie_hw_network_servers_get_by_url : Ttiepie_hw_network_servers_get_by_url = nil;
  tiepie_hw_network_servers_set_callback_added : Ttiepie_hw_network_servers_set_callback_added = nil;
  tiepie_hw_object_close : Ttiepie_hw_object_close = nil;
  tiepie_hw_object_is_removed : Ttiepie_hw_object_is_removed = nil;
  tiepie_hw_object_get_interfaces : Ttiepie_hw_object_get_interfaces = nil;
  tiepie_hw_object_set_event_callback : Ttiepie_hw_object_set_event_callback = nil;
  tiepie_hw_device_get_calibration_date : Ttiepie_hw_device_get_calibration_date = nil;
  tiepie_hw_device_get_serial_number : Ttiepie_hw_device_get_serial_number = nil;
  tiepie_hw_device_get_ip_address : Ttiepie_hw_device_get_ip_address = nil;
  tiepie_hw_device_get_ip_port : Ttiepie_hw_device_get_ip_port = nil;
  tiepie_hw_device_get_product_id : Ttiepie_hw_device_get_product_id = nil;
  tiepie_hw_device_get_type : Ttiepie_hw_device_get_type = nil;
  tiepie_hw_device_get_name : Ttiepie_hw_device_get_name = nil;
  tiepie_hw_device_get_name_short : Ttiepie_hw_device_get_name_short = nil;
  tiepie_hw_device_get_name_shortest : Ttiepie_hw_device_get_name_shortest = nil;
  tiepie_hw_device_has_wireless_trigger_module : Ttiepie_hw_device_has_wireless_trigger_module = nil;
  tiepie_hw_device_has_battery : Ttiepie_hw_device_has_battery = nil;
  tiepie_hw_device_has_battery_ex : Ttiepie_hw_device_has_battery_ex = nil;
  tiepie_hw_device_get_battery_charge : Ttiepie_hw_device_get_battery_charge = nil;
  tiepie_hw_device_get_battery_charge_ex : Ttiepie_hw_device_get_battery_charge_ex = nil;
  tiepie_hw_device_get_battery_time_to_empty : Ttiepie_hw_device_get_battery_time_to_empty = nil;
  tiepie_hw_device_get_battery_time_to_empty_ex : Ttiepie_hw_device_get_battery_time_to_empty_ex = nil;
  tiepie_hw_device_get_battery_time_to_full : Ttiepie_hw_device_get_battery_time_to_full = nil;
  tiepie_hw_device_get_battery_time_to_full_ex : Ttiepie_hw_device_get_battery_time_to_full_ex = nil;
  tiepie_hw_device_is_battery_charger_connected : Ttiepie_hw_device_is_battery_charger_connected = nil;
  tiepie_hw_device_is_battery_charger_connected_ex : Ttiepie_hw_device_is_battery_charger_connected_ex = nil;
  tiepie_hw_device_is_battery_charging : Ttiepie_hw_device_is_battery_charging = nil;
  tiepie_hw_device_is_battery_charging_ex : Ttiepie_hw_device_is_battery_charging_ex = nil;
  tiepie_hw_device_is_battery_broken : Ttiepie_hw_device_is_battery_broken = nil;
  tiepie_hw_device_is_battery_broken_ex : Ttiepie_hw_device_is_battery_broken_ex = nil;
  tiepie_hw_device_trigger_get_input_count : Ttiepie_hw_device_trigger_get_input_count = nil;
  tiepie_hw_device_trigger_get_input_index_by_id : Ttiepie_hw_device_trigger_get_input_index_by_id = nil;
  tiepie_hw_oscilloscope_trigger_input_is_triggered : Ttiepie_hw_oscilloscope_trigger_input_is_triggered = nil;
  tiepie_hw_device_trigger_input_get_enabled : Ttiepie_hw_device_trigger_input_get_enabled = nil;
  tiepie_hw_device_trigger_input_set_enabled : Ttiepie_hw_device_trigger_input_set_enabled = nil;
  tiepie_hw_device_trigger_input_get_kinds : Ttiepie_hw_device_trigger_input_get_kinds = nil;
  tiepie_hw_oscilloscope_trigger_input_get_kinds_ex : Ttiepie_hw_oscilloscope_trigger_input_get_kinds_ex = nil;
  tiepie_hw_device_trigger_input_get_kind : Ttiepie_hw_device_trigger_input_get_kind = nil;
  tiepie_hw_device_trigger_input_set_kind : Ttiepie_hw_device_trigger_input_set_kind = nil;
  tiepie_hw_device_trigger_input_is_available : Ttiepie_hw_device_trigger_input_is_available = nil;
  tiepie_hw_oscilloscope_trigger_input_is_available_ex : Ttiepie_hw_oscilloscope_trigger_input_is_available_ex = nil;
  tiepie_hw_device_trigger_input_get_id : Ttiepie_hw_device_trigger_input_get_id = nil;
  tiepie_hw_device_trigger_input_get_name : Ttiepie_hw_device_trigger_input_get_name = nil;
  tiepie_hw_device_trigger_get_output_count : Ttiepie_hw_device_trigger_get_output_count = nil;
  tiepie_hw_device_trigger_get_output_index_by_id : Ttiepie_hw_device_trigger_get_output_index_by_id = nil;
  tiepie_hw_device_trigger_output_get_enabled : Ttiepie_hw_device_trigger_output_get_enabled = nil;
  tiepie_hw_device_trigger_output_set_enabled : Ttiepie_hw_device_trigger_output_set_enabled = nil;
  tiepie_hw_device_trigger_output_get_events : Ttiepie_hw_device_trigger_output_get_events = nil;
  tiepie_hw_device_trigger_output_get_event : Ttiepie_hw_device_trigger_output_get_event = nil;
  tiepie_hw_device_trigger_output_set_event : Ttiepie_hw_device_trigger_output_set_event = nil;
  tiepie_hw_device_trigger_output_get_id : Ttiepie_hw_device_trigger_output_get_id = nil;
  tiepie_hw_device_trigger_output_get_name : Ttiepie_hw_device_trigger_output_get_name = nil;
  tiepie_hw_device_trigger_output_trigger : Ttiepie_hw_device_trigger_output_trigger = nil;
  tiepie_hw_oscilloscope_get_channel_count : Ttiepie_hw_oscilloscope_get_channel_count = nil;
  tiepie_hw_oscilloscope_channel_is_available : Ttiepie_hw_oscilloscope_channel_is_available = nil;
  tiepie_hw_oscilloscope_channel_is_available_ex : Ttiepie_hw_oscilloscope_channel_is_available_ex = nil;
  tiepie_hw_oscilloscope_channel_get_connector_type : Ttiepie_hw_oscilloscope_channel_get_connector_type = nil;
  tiepie_hw_oscilloscope_channel_is_differential : Ttiepie_hw_oscilloscope_channel_is_differential = nil;
  tiepie_hw_oscilloscope_channel_is_isolated : Ttiepie_hw_oscilloscope_channel_is_isolated = nil;
  tiepie_hw_oscilloscope_channel_get_impedance : Ttiepie_hw_oscilloscope_channel_get_impedance = nil;
  tiepie_hw_oscilloscope_channel_get_bandwidths : Ttiepie_hw_oscilloscope_channel_get_bandwidths = nil;
  tiepie_hw_oscilloscope_channel_get_bandwidth : Ttiepie_hw_oscilloscope_channel_get_bandwidth = nil;
  tiepie_hw_oscilloscope_channel_set_bandwidth : Ttiepie_hw_oscilloscope_channel_set_bandwidth = nil;
  tiepie_hw_oscilloscope_channel_get_couplings : Ttiepie_hw_oscilloscope_channel_get_couplings = nil;
  tiepie_hw_oscilloscope_channel_get_coupling : Ttiepie_hw_oscilloscope_channel_get_coupling = nil;
  tiepie_hw_oscilloscope_channel_set_coupling : Ttiepie_hw_oscilloscope_channel_set_coupling = nil;
  tiepie_hw_oscilloscope_channel_get_enabled : Ttiepie_hw_oscilloscope_channel_get_enabled = nil;
  tiepie_hw_oscilloscope_channel_set_enabled : Ttiepie_hw_oscilloscope_channel_set_enabled = nil;
  tiepie_hw_oscilloscope_channel_get_auto_ranging : Ttiepie_hw_oscilloscope_channel_get_auto_ranging = nil;
  tiepie_hw_oscilloscope_channel_set_auto_ranging : Ttiepie_hw_oscilloscope_channel_set_auto_ranging = nil;
  tiepie_hw_oscilloscope_channel_get_ranges : Ttiepie_hw_oscilloscope_channel_get_ranges = nil;
  tiepie_hw_oscilloscope_channel_get_ranges_ex : Ttiepie_hw_oscilloscope_channel_get_ranges_ex = nil;
  tiepie_hw_oscilloscope_channel_get_range : Ttiepie_hw_oscilloscope_channel_get_range = nil;
  tiepie_hw_oscilloscope_channel_set_range : Ttiepie_hw_oscilloscope_channel_set_range = nil;
  tiepie_hw_oscilloscope_channel_has_safeground : Ttiepie_hw_oscilloscope_channel_has_safeground = nil;
  tiepie_hw_oscilloscope_channel_get_safeground_enabled : Ttiepie_hw_oscilloscope_channel_get_safeground_enabled = nil;
  tiepie_hw_oscilloscope_channel_set_safeground_enabled : Ttiepie_hw_oscilloscope_channel_set_safeground_enabled = nil;
  tiepie_hw_oscilloscope_channel_get_safeground_threshold_min : Ttiepie_hw_oscilloscope_channel_get_safeground_threshold_min = nil;
  tiepie_hw_oscilloscope_channel_get_safeground_threshold_max : Ttiepie_hw_oscilloscope_channel_get_safeground_threshold_max = nil;
  tiepie_hw_oscilloscope_channel_get_safeground_threshold : Ttiepie_hw_oscilloscope_channel_get_safeground_threshold = nil;
  tiepie_hw_oscilloscope_channel_set_safeground_threshold : Ttiepie_hw_oscilloscope_channel_set_safeground_threshold = nil;
  tiepie_hw_oscilloscope_channel_verify_safeground_threshold : Ttiepie_hw_oscilloscope_channel_verify_safeground_threshold = nil;
  tiepie_hw_oscilloscope_channel_has_trigger : Ttiepie_hw_oscilloscope_channel_has_trigger = nil;
  tiepie_hw_oscilloscope_channel_has_trigger_ex : Ttiepie_hw_oscilloscope_channel_has_trigger_ex = nil;
  tiepie_hw_oscilloscope_channel_trigger_is_available : Ttiepie_hw_oscilloscope_channel_trigger_is_available = nil;
  tiepie_hw_oscilloscope_channel_trigger_is_available_ex : Ttiepie_hw_oscilloscope_channel_trigger_is_available_ex = nil;
  tiepie_hw_oscilloscope_channel_trigger_is_triggered : Ttiepie_hw_oscilloscope_channel_trigger_is_triggered = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_enabled : Ttiepie_hw_oscilloscope_channel_trigger_get_enabled = nil;
  tiepie_hw_oscilloscope_channel_trigger_set_enabled : Ttiepie_hw_oscilloscope_channel_trigger_set_enabled = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_kinds : Ttiepie_hw_oscilloscope_channel_trigger_get_kinds = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_kinds_ex : Ttiepie_hw_oscilloscope_channel_trigger_get_kinds_ex = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_kind : Ttiepie_hw_oscilloscope_channel_trigger_get_kind = nil;
  tiepie_hw_oscilloscope_channel_trigger_set_kind : Ttiepie_hw_oscilloscope_channel_trigger_set_kind = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_level_modes : Ttiepie_hw_oscilloscope_channel_trigger_get_level_modes = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_level_mode : Ttiepie_hw_oscilloscope_channel_trigger_get_level_mode = nil;
  tiepie_hw_oscilloscope_channel_trigger_set_level_mode : Ttiepie_hw_oscilloscope_channel_trigger_set_level_mode = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_level_count : Ttiepie_hw_oscilloscope_channel_trigger_get_level_count = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_level : Ttiepie_hw_oscilloscope_channel_trigger_get_level = nil;
  tiepie_hw_oscilloscope_channel_trigger_set_level : Ttiepie_hw_oscilloscope_channel_trigger_set_level = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count : Ttiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_hysteresis : Ttiepie_hw_oscilloscope_channel_trigger_get_hysteresis = nil;
  tiepie_hw_oscilloscope_channel_trigger_set_hysteresis : Ttiepie_hw_oscilloscope_channel_trigger_set_hysteresis = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_conditions : Ttiepie_hw_oscilloscope_channel_trigger_get_conditions = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_conditions_ex : Ttiepie_hw_oscilloscope_channel_trigger_get_conditions_ex = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_condition : Ttiepie_hw_oscilloscope_channel_trigger_get_condition = nil;
  tiepie_hw_oscilloscope_channel_trigger_set_condition : Ttiepie_hw_oscilloscope_channel_trigger_set_condition = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_time_count : Ttiepie_hw_oscilloscope_channel_trigger_get_time_count = nil;
  tiepie_hw_oscilloscope_channel_trigger_get_time : Ttiepie_hw_oscilloscope_channel_trigger_get_time = nil;
  tiepie_hw_oscilloscope_channel_trigger_set_time : Ttiepie_hw_oscilloscope_channel_trigger_set_time = nil;
  tiepie_hw_oscilloscope_channel_trigger_verify_time : Ttiepie_hw_oscilloscope_channel_trigger_verify_time = nil;
  tiepie_hw_oscilloscope_channel_trigger_verify_time_ex : Ttiepie_hw_oscilloscope_channel_trigger_verify_time_ex = nil;
  tiepie_hw_oscilloscope_is_demo : Ttiepie_hw_oscilloscope_is_demo = nil;
  tiepie_hw_oscilloscope_channel_demo_get_signal : Ttiepie_hw_oscilloscope_channel_demo_get_signal = nil;
  tiepie_hw_oscilloscope_channel_demo_set_signal : Ttiepie_hw_oscilloscope_channel_demo_set_signal = nil;
  tiepie_hw_oscilloscope_channel_demo_has_amplitude : Ttiepie_hw_oscilloscope_channel_demo_has_amplitude = nil;
  tiepie_hw_oscilloscope_channel_demo_get_amplitude : Ttiepie_hw_oscilloscope_channel_demo_get_amplitude = nil;
  tiepie_hw_oscilloscope_channel_demo_set_amplitude : Ttiepie_hw_oscilloscope_channel_demo_set_amplitude = nil;
  tiepie_hw_oscilloscope_channel_demo_has_frequency : Ttiepie_hw_oscilloscope_channel_demo_has_frequency = nil;
  tiepie_hw_oscilloscope_channel_demo_get_frequency : Ttiepie_hw_oscilloscope_channel_demo_get_frequency = nil;
  tiepie_hw_oscilloscope_channel_demo_set_frequency : Ttiepie_hw_oscilloscope_channel_demo_set_frequency = nil;
  tiepie_hw_oscilloscope_channel_demo_has_offset : Ttiepie_hw_oscilloscope_channel_demo_has_offset = nil;
  tiepie_hw_oscilloscope_channel_demo_get_offset : Ttiepie_hw_oscilloscope_channel_demo_get_offset = nil;
  tiepie_hw_oscilloscope_channel_demo_set_offset : Ttiepie_hw_oscilloscope_channel_demo_set_offset = nil;
  tiepie_hw_oscilloscope_channel_demo_has_symmetry : Ttiepie_hw_oscilloscope_channel_demo_has_symmetry = nil;
  tiepie_hw_oscilloscope_channel_demo_get_symmetry : Ttiepie_hw_oscilloscope_channel_demo_get_symmetry = nil;
  tiepie_hw_oscilloscope_channel_demo_set_symmetry : Ttiepie_hw_oscilloscope_channel_demo_set_symmetry = nil;
  tiepie_hw_oscilloscope_channel_demo_has_phase : Ttiepie_hw_oscilloscope_channel_demo_has_phase = nil;
  tiepie_hw_oscilloscope_channel_demo_get_phase : Ttiepie_hw_oscilloscope_channel_demo_get_phase = nil;
  tiepie_hw_oscilloscope_channel_demo_set_phase : Ttiepie_hw_oscilloscope_channel_demo_set_phase = nil;
  tiepie_hw_oscilloscope_channel_demo_has_noise_enabled : Ttiepie_hw_oscilloscope_channel_demo_has_noise_enabled = nil;
  tiepie_hw_oscilloscope_channel_demo_get_noise_enabled : Ttiepie_hw_oscilloscope_channel_demo_get_noise_enabled = nil;
  tiepie_hw_oscilloscope_channel_demo_set_noise_enabled : Ttiepie_hw_oscilloscope_channel_demo_set_noise_enabled = nil;
  tiepie_hw_oscilloscope_get_data : Ttiepie_hw_oscilloscope_get_data = nil;
  tiepie_hw_oscilloscope_get_data_1ch : Ttiepie_hw_oscilloscope_get_data_1ch = nil;
  tiepie_hw_oscilloscope_get_data_2ch : Ttiepie_hw_oscilloscope_get_data_2ch = nil;
  tiepie_hw_oscilloscope_get_data_3ch : Ttiepie_hw_oscilloscope_get_data_3ch = nil;
  tiepie_hw_oscilloscope_get_data_4ch : Ttiepie_hw_oscilloscope_get_data_4ch = nil;
  tiepie_hw_oscilloscope_get_data_5ch : Ttiepie_hw_oscilloscope_get_data_5ch = nil;
  tiepie_hw_oscilloscope_get_data_6ch : Ttiepie_hw_oscilloscope_get_data_6ch = nil;
  tiepie_hw_oscilloscope_get_data_7ch : Ttiepie_hw_oscilloscope_get_data_7ch = nil;
  tiepie_hw_oscilloscope_get_data_8ch : Ttiepie_hw_oscilloscope_get_data_8ch = nil;
  tiepie_hw_oscilloscope_get_valid_pre_sample_count : Ttiepie_hw_oscilloscope_get_valid_pre_sample_count = nil;
  tiepie_hw_oscilloscope_channel_get_data_value_range : Ttiepie_hw_oscilloscope_channel_get_data_value_range = nil;
  tiepie_hw_oscilloscope_channel_get_data_value_min : Ttiepie_hw_oscilloscope_channel_get_data_value_min = nil;
  tiepie_hw_oscilloscope_channel_get_data_value_max : Ttiepie_hw_oscilloscope_channel_get_data_value_max = nil;
  tiepie_hw_oscilloscope_get_data_raw : Ttiepie_hw_oscilloscope_get_data_raw = nil;
  tiepie_hw_oscilloscope_get_data_raw_1ch : Ttiepie_hw_oscilloscope_get_data_raw_1ch = nil;
  tiepie_hw_oscilloscope_get_data_raw_2ch : Ttiepie_hw_oscilloscope_get_data_raw_2ch = nil;
  tiepie_hw_oscilloscope_get_data_raw_3ch : Ttiepie_hw_oscilloscope_get_data_raw_3ch = nil;
  tiepie_hw_oscilloscope_get_data_raw_4ch : Ttiepie_hw_oscilloscope_get_data_raw_4ch = nil;
  tiepie_hw_oscilloscope_get_data_raw_5ch : Ttiepie_hw_oscilloscope_get_data_raw_5ch = nil;
  tiepie_hw_oscilloscope_get_data_raw_6ch : Ttiepie_hw_oscilloscope_get_data_raw_6ch = nil;
  tiepie_hw_oscilloscope_get_data_raw_7ch : Ttiepie_hw_oscilloscope_get_data_raw_7ch = nil;
  tiepie_hw_oscilloscope_get_data_raw_8ch : Ttiepie_hw_oscilloscope_get_data_raw_8ch = nil;
  tiepie_hw_oscilloscope_channel_get_data_raw_type : Ttiepie_hw_oscilloscope_channel_get_data_raw_type = nil;
  tiepie_hw_oscilloscope_channel_get_data_raw_value_range : Ttiepie_hw_oscilloscope_channel_get_data_raw_value_range = nil;
  tiepie_hw_oscilloscope_channel_get_data_raw_value_min : Ttiepie_hw_oscilloscope_channel_get_data_raw_value_min = nil;
  tiepie_hw_oscilloscope_channel_get_data_raw_value_zero : Ttiepie_hw_oscilloscope_channel_get_data_raw_value_zero = nil;
  tiepie_hw_oscilloscope_channel_get_data_raw_value_max : Ttiepie_hw_oscilloscope_channel_get_data_raw_value_max = nil;
  tiepie_hw_oscilloscope_channel_is_range_max_reachable : Ttiepie_hw_oscilloscope_channel_is_range_max_reachable = nil;
  tiepie_hw_oscilloscope_start : Ttiepie_hw_oscilloscope_start = nil;
  tiepie_hw_oscilloscope_stop : Ttiepie_hw_oscilloscope_stop = nil;
  tiepie_hw_oscilloscope_force_trigger : Ttiepie_hw_oscilloscope_force_trigger = nil;
  tiepie_hw_oscilloscope_get_measure_modes : Ttiepie_hw_oscilloscope_get_measure_modes = nil;
  tiepie_hw_oscilloscope_get_measure_mode : Ttiepie_hw_oscilloscope_get_measure_mode = nil;
  tiepie_hw_oscilloscope_set_measure_mode : Ttiepie_hw_oscilloscope_set_measure_mode = nil;
  tiepie_hw_oscilloscope_is_running : Ttiepie_hw_oscilloscope_is_running = nil;
  tiepie_hw_oscilloscope_is_triggered : Ttiepie_hw_oscilloscope_is_triggered = nil;
  tiepie_hw_oscilloscope_is_timeout_triggered : Ttiepie_hw_oscilloscope_is_timeout_triggered = nil;
  tiepie_hw_oscilloscope_is_force_triggered : Ttiepie_hw_oscilloscope_is_force_triggered = nil;
  tiepie_hw_oscilloscope_is_data_ready : Ttiepie_hw_oscilloscope_is_data_ready = nil;
  tiepie_hw_oscilloscope_is_data_overflow : Ttiepie_hw_oscilloscope_is_data_overflow = nil;
  tiepie_hw_oscilloscope_get_auto_resolution_modes : Ttiepie_hw_oscilloscope_get_auto_resolution_modes = nil;
  tiepie_hw_oscilloscope_get_auto_resolution_mode : Ttiepie_hw_oscilloscope_get_auto_resolution_mode = nil;
  tiepie_hw_oscilloscope_set_auto_resolution_mode : Ttiepie_hw_oscilloscope_set_auto_resolution_mode = nil;
  tiepie_hw_oscilloscope_get_resolutions : Ttiepie_hw_oscilloscope_get_resolutions = nil;
  tiepie_hw_oscilloscope_get_resolution : Ttiepie_hw_oscilloscope_get_resolution = nil;
  tiepie_hw_oscilloscope_set_resolution : Ttiepie_hw_oscilloscope_set_resolution = nil;
  tiepie_hw_oscilloscope_is_resolution_enhanced : Ttiepie_hw_oscilloscope_is_resolution_enhanced = nil;
  tiepie_hw_oscilloscope_is_resolution_enhanced_ex : Ttiepie_hw_oscilloscope_is_resolution_enhanced_ex = nil;
  tiepie_hw_oscilloscope_get_clock_sources : Ttiepie_hw_oscilloscope_get_clock_sources = nil;
  tiepie_hw_oscilloscope_get_clock_source : Ttiepie_hw_oscilloscope_get_clock_source = nil;
  tiepie_hw_oscilloscope_set_clock_source : Ttiepie_hw_oscilloscope_set_clock_source = nil;
  tiepie_hw_oscilloscope_get_clock_source_frequencies : Ttiepie_hw_oscilloscope_get_clock_source_frequencies = nil;
  tiepie_hw_oscilloscope_get_clock_source_frequencies_ex : Ttiepie_hw_oscilloscope_get_clock_source_frequencies_ex = nil;
  tiepie_hw_oscilloscope_get_clock_source_frequency : Ttiepie_hw_oscilloscope_get_clock_source_frequency = nil;
  tiepie_hw_oscilloscope_set_clock_source_frequency : Ttiepie_hw_oscilloscope_set_clock_source_frequency = nil;
  tiepie_hw_oscilloscope_get_clock_outputs : Ttiepie_hw_oscilloscope_get_clock_outputs = nil;
  tiepie_hw_oscilloscope_get_clock_output : Ttiepie_hw_oscilloscope_get_clock_output = nil;
  tiepie_hw_oscilloscope_set_clock_output : Ttiepie_hw_oscilloscope_set_clock_output = nil;
  tiepie_hw_oscilloscope_get_clock_output_frequencies : Ttiepie_hw_oscilloscope_get_clock_output_frequencies = nil;
  tiepie_hw_oscilloscope_get_clock_output_frequencies_ex : Ttiepie_hw_oscilloscope_get_clock_output_frequencies_ex = nil;
  tiepie_hw_oscilloscope_get_clock_output_frequency : Ttiepie_hw_oscilloscope_get_clock_output_frequency = nil;
  tiepie_hw_oscilloscope_set_clock_output_frequency : Ttiepie_hw_oscilloscope_set_clock_output_frequency = nil;
  tiepie_hw_oscilloscope_get_sample_rate_max : Ttiepie_hw_oscilloscope_get_sample_rate_max = nil;
  tiepie_hw_oscilloscope_get_sample_rate : Ttiepie_hw_oscilloscope_get_sample_rate = nil;
  tiepie_hw_oscilloscope_set_sample_rate : Ttiepie_hw_oscilloscope_set_sample_rate = nil;
  tiepie_hw_oscilloscope_verify_sample_rate : Ttiepie_hw_oscilloscope_verify_sample_rate = nil;
  tiepie_hw_oscilloscope_verify_sample_rate_ex : Ttiepie_hw_oscilloscope_verify_sample_rate_ex = nil;
  tiepie_hw_oscilloscope_verify_sample_rates_ex : Ttiepie_hw_oscilloscope_verify_sample_rates_ex = nil;
  tiepie_hw_oscilloscope_get_record_length_max : Ttiepie_hw_oscilloscope_get_record_length_max = nil;
  tiepie_hw_oscilloscope_get_record_length_max_ex : Ttiepie_hw_oscilloscope_get_record_length_max_ex = nil;
  tiepie_hw_oscilloscope_get_record_length : Ttiepie_hw_oscilloscope_get_record_length = nil;
  tiepie_hw_oscilloscope_set_record_length : Ttiepie_hw_oscilloscope_set_record_length = nil;
  tiepie_hw_oscilloscope_verify_record_length : Ttiepie_hw_oscilloscope_verify_record_length = nil;
  tiepie_hw_oscilloscope_verify_record_length_ex : Ttiepie_hw_oscilloscope_verify_record_length_ex = nil;
  tiepie_hw_oscilloscope_get_pre_sample_ratio : Ttiepie_hw_oscilloscope_get_pre_sample_ratio = nil;
  tiepie_hw_oscilloscope_set_pre_sample_ratio : Ttiepie_hw_oscilloscope_set_pre_sample_ratio = nil;
  tiepie_hw_oscilloscope_get_segment_count_max : Ttiepie_hw_oscilloscope_get_segment_count_max = nil;
  tiepie_hw_oscilloscope_get_segment_count_max_ex : Ttiepie_hw_oscilloscope_get_segment_count_max_ex = nil;
  tiepie_hw_oscilloscope_get_segment_count : Ttiepie_hw_oscilloscope_get_segment_count = nil;
  tiepie_hw_oscilloscope_set_segment_count : Ttiepie_hw_oscilloscope_set_segment_count = nil;
  tiepie_hw_oscilloscope_verify_segment_count : Ttiepie_hw_oscilloscope_verify_segment_count = nil;
  tiepie_hw_oscilloscope_verify_segment_count_ex : Ttiepie_hw_oscilloscope_verify_segment_count_ex = nil;
  tiepie_hw_oscilloscope_has_trigger : Ttiepie_hw_oscilloscope_has_trigger = nil;
  tiepie_hw_oscilloscope_has_trigger_ex : Ttiepie_hw_oscilloscope_has_trigger_ex = nil;
  tiepie_hw_oscilloscope_trigger_get_timeout : Ttiepie_hw_oscilloscope_trigger_get_timeout = nil;
  tiepie_hw_oscilloscope_trigger_set_timeout : Ttiepie_hw_oscilloscope_trigger_set_timeout = nil;
  tiepie_hw_oscilloscope_trigger_verify_timeout : Ttiepie_hw_oscilloscope_trigger_verify_timeout = nil;
  tiepie_hw_oscilloscope_trigger_verify_timeout_ex : Ttiepie_hw_oscilloscope_trigger_verify_timeout_ex = nil;
  tiepie_hw_oscilloscope_trigger_has_delay : Ttiepie_hw_oscilloscope_trigger_has_delay = nil;
  tiepie_hw_oscilloscope_trigger_has_delay_ex : Ttiepie_hw_oscilloscope_trigger_has_delay_ex = nil;
  tiepie_hw_oscilloscope_trigger_get_delay_max : Ttiepie_hw_oscilloscope_trigger_get_delay_max = nil;
  tiepie_hw_oscilloscope_trigger_get_delay_max_ex : Ttiepie_hw_oscilloscope_trigger_get_delay_max_ex = nil;
  tiepie_hw_oscilloscope_trigger_get_delay : Ttiepie_hw_oscilloscope_trigger_get_delay = nil;
  tiepie_hw_oscilloscope_trigger_set_delay : Ttiepie_hw_oscilloscope_trigger_set_delay = nil;
  tiepie_hw_oscilloscope_trigger_verify_delay : Ttiepie_hw_oscilloscope_trigger_verify_delay = nil;
  tiepie_hw_oscilloscope_trigger_verify_delay_ex : Ttiepie_hw_oscilloscope_trigger_verify_delay_ex = nil;
  tiepie_hw_oscilloscope_has_presamples_valid : Ttiepie_hw_oscilloscope_has_presamples_valid = nil;
  tiepie_hw_oscilloscope_has_presamples_valid_ex : Ttiepie_hw_oscilloscope_has_presamples_valid_ex = nil;
  tiepie_hw_oscilloscope_get_presamples_valid : Ttiepie_hw_oscilloscope_get_presamples_valid = nil;
  tiepie_hw_oscilloscope_set_presamples_valid : Ttiepie_hw_oscilloscope_set_presamples_valid = nil;
  tiepie_hw_oscilloscope_has_sureconnect : Ttiepie_hw_oscilloscope_has_sureconnect = nil;
  tiepie_hw_oscilloscope_channel_has_sureconnect : Ttiepie_hw_oscilloscope_channel_has_sureconnect = nil;
  tiepie_hw_oscilloscope_start_sureconnect : Ttiepie_hw_oscilloscope_start_sureconnect = nil;
  tiepie_hw_oscilloscope_start_sureconnect_ex : Ttiepie_hw_oscilloscope_start_sureconnect_ex = nil;
  tiepie_hw_oscilloscope_is_sureconnect_completed : Ttiepie_hw_oscilloscope_is_sureconnect_completed = nil;
  tiepie_hw_oscilloscope_get_sureconnect_data : Ttiepie_hw_oscilloscope_get_sureconnect_data = nil;
  tiepie_hw_generator_get_connector_type : Ttiepie_hw_generator_get_connector_type = nil;
  tiepie_hw_generator_is_differential : Ttiepie_hw_generator_is_differential = nil;
  tiepie_hw_generator_get_impedance : Ttiepie_hw_generator_get_impedance = nil;
  tiepie_hw_generator_get_resolution : Ttiepie_hw_generator_get_resolution = nil;
  tiepie_hw_generator_get_output_value_min : Ttiepie_hw_generator_get_output_value_min = nil;
  tiepie_hw_generator_get_output_value_max : Ttiepie_hw_generator_get_output_value_max = nil;
  tiepie_hw_generator_get_output_value_min_max : Ttiepie_hw_generator_get_output_value_min_max = nil;
  tiepie_hw_generator_is_controllable : Ttiepie_hw_generator_is_controllable = nil;
  tiepie_hw_generator_is_running : Ttiepie_hw_generator_is_running = nil;
  tiepie_hw_generator_get_status : Ttiepie_hw_generator_get_status = nil;
  tiepie_hw_generator_get_output_enable : Ttiepie_hw_generator_get_output_enable = nil;
  tiepie_hw_generator_set_output_enable : Ttiepie_hw_generator_set_output_enable = nil;
  tiepie_hw_generator_has_output_invert : Ttiepie_hw_generator_has_output_invert = nil;
  tiepie_hw_generator_get_output_invert : Ttiepie_hw_generator_get_output_invert = nil;
  tiepie_hw_generator_set_output_invert : Ttiepie_hw_generator_set_output_invert = nil;
  tiepie_hw_generator_start : Ttiepie_hw_generator_start = nil;
  tiepie_hw_generator_stop : Ttiepie_hw_generator_stop = nil;
  tiepie_hw_generator_get_signal_types : Ttiepie_hw_generator_get_signal_types = nil;
  tiepie_hw_generator_get_signal_type : Ttiepie_hw_generator_get_signal_type = nil;
  tiepie_hw_generator_set_signal_type : Ttiepie_hw_generator_set_signal_type = nil;
  tiepie_hw_generator_has_amplitude : Ttiepie_hw_generator_has_amplitude = nil;
  tiepie_hw_generator_has_amplitude_ex : Ttiepie_hw_generator_has_amplitude_ex = nil;
  tiepie_hw_generator_get_amplitude_min : Ttiepie_hw_generator_get_amplitude_min = nil;
  tiepie_hw_generator_get_amplitude_max : Ttiepie_hw_generator_get_amplitude_max = nil;
  tiepie_hw_generator_get_amplitude_min_max_ex : Ttiepie_hw_generator_get_amplitude_min_max_ex = nil;
  tiepie_hw_generator_get_amplitude : Ttiepie_hw_generator_get_amplitude = nil;
  tiepie_hw_generator_set_amplitude : Ttiepie_hw_generator_set_amplitude = nil;
  tiepie_hw_generator_verify_amplitude : Ttiepie_hw_generator_verify_amplitude = nil;
  tiepie_hw_generator_verify_amplitude_ex : Ttiepie_hw_generator_verify_amplitude_ex = nil;
  tiepie_hw_generator_get_amplitude_ranges : Ttiepie_hw_generator_get_amplitude_ranges = nil;
  tiepie_hw_generator_get_amplitude_range : Ttiepie_hw_generator_get_amplitude_range = nil;
  tiepie_hw_generator_set_amplitude_range : Ttiepie_hw_generator_set_amplitude_range = nil;
  tiepie_hw_generator_get_amplitude_auto_ranging : Ttiepie_hw_generator_get_amplitude_auto_ranging = nil;
  tiepie_hw_generator_set_amplitude_auto_ranging : Ttiepie_hw_generator_set_amplitude_auto_ranging = nil;
  tiepie_hw_generator_has_offset : Ttiepie_hw_generator_has_offset = nil;
  tiepie_hw_generator_has_offset_ex : Ttiepie_hw_generator_has_offset_ex = nil;
  tiepie_hw_generator_get_offset_min : Ttiepie_hw_generator_get_offset_min = nil;
  tiepie_hw_generator_get_offset_max : Ttiepie_hw_generator_get_offset_max = nil;
  tiepie_hw_generator_get_offset_min_max_ex : Ttiepie_hw_generator_get_offset_min_max_ex = nil;
  tiepie_hw_generator_get_offset : Ttiepie_hw_generator_get_offset = nil;
  tiepie_hw_generator_set_offset : Ttiepie_hw_generator_set_offset = nil;
  tiepie_hw_generator_verify_offset : Ttiepie_hw_generator_verify_offset = nil;
  tiepie_hw_generator_verify_offset_ex : Ttiepie_hw_generator_verify_offset_ex = nil;
  tiepie_hw_generator_get_frequency_modes : Ttiepie_hw_generator_get_frequency_modes = nil;
  tiepie_hw_generator_get_frequency_modes_ex : Ttiepie_hw_generator_get_frequency_modes_ex = nil;
  tiepie_hw_generator_get_frequency_mode : Ttiepie_hw_generator_get_frequency_mode = nil;
  tiepie_hw_generator_set_frequency_mode : Ttiepie_hw_generator_set_frequency_mode = nil;
  tiepie_hw_generator_has_frequency : Ttiepie_hw_generator_has_frequency = nil;
  tiepie_hw_generator_has_frequency_ex : Ttiepie_hw_generator_has_frequency_ex = nil;
  tiepie_hw_generator_get_frequency_min : Ttiepie_hw_generator_get_frequency_min = nil;
  tiepie_hw_generator_get_frequency_max : Ttiepie_hw_generator_get_frequency_max = nil;
  tiepie_hw_generator_get_frequency_min_max : Ttiepie_hw_generator_get_frequency_min_max = nil;
  tiepie_hw_generator_get_frequency_min_max_ex : Ttiepie_hw_generator_get_frequency_min_max_ex = nil;
  tiepie_hw_generator_get_frequency : Ttiepie_hw_generator_get_frequency = nil;
  tiepie_hw_generator_set_frequency : Ttiepie_hw_generator_set_frequency = nil;
  tiepie_hw_generator_verify_frequency : Ttiepie_hw_generator_verify_frequency = nil;
  tiepie_hw_generator_verify_frequency_ex : Ttiepie_hw_generator_verify_frequency_ex = nil;
  tiepie_hw_generator_has_phase : Ttiepie_hw_generator_has_phase = nil;
  tiepie_hw_generator_has_phase_ex : Ttiepie_hw_generator_has_phase_ex = nil;
  tiepie_hw_generator_get_phase_min : Ttiepie_hw_generator_get_phase_min = nil;
  tiepie_hw_generator_get_phase_max : Ttiepie_hw_generator_get_phase_max = nil;
  tiepie_hw_generator_get_phase_min_max_ex : Ttiepie_hw_generator_get_phase_min_max_ex = nil;
  tiepie_hw_generator_get_phase : Ttiepie_hw_generator_get_phase = nil;
  tiepie_hw_generator_set_phase : Ttiepie_hw_generator_set_phase = nil;
  tiepie_hw_generator_verify_phase : Ttiepie_hw_generator_verify_phase = nil;
  tiepie_hw_generator_verify_phase_ex : Ttiepie_hw_generator_verify_phase_ex = nil;
  tiepie_hw_generator_has_symmetry : Ttiepie_hw_generator_has_symmetry = nil;
  tiepie_hw_generator_has_symmetry_ex : Ttiepie_hw_generator_has_symmetry_ex = nil;
  tiepie_hw_generator_get_symmetry_min : Ttiepie_hw_generator_get_symmetry_min = nil;
  tiepie_hw_generator_get_symmetry_max : Ttiepie_hw_generator_get_symmetry_max = nil;
  tiepie_hw_generator_get_symmetry_min_max_ex : Ttiepie_hw_generator_get_symmetry_min_max_ex = nil;
  tiepie_hw_generator_get_symmetry : Ttiepie_hw_generator_get_symmetry = nil;
  tiepie_hw_generator_set_symmetry : Ttiepie_hw_generator_set_symmetry = nil;
  tiepie_hw_generator_verify_symmetry : Ttiepie_hw_generator_verify_symmetry = nil;
  tiepie_hw_generator_verify_symmetry_ex : Ttiepie_hw_generator_verify_symmetry_ex = nil;
  tiepie_hw_generator_has_width : Ttiepie_hw_generator_has_width = nil;
  tiepie_hw_generator_has_width_ex : Ttiepie_hw_generator_has_width_ex = nil;
  tiepie_hw_generator_get_width_min : Ttiepie_hw_generator_get_width_min = nil;
  tiepie_hw_generator_get_width_max : Ttiepie_hw_generator_get_width_max = nil;
  tiepie_hw_generator_get_width_min_max_ex : Ttiepie_hw_generator_get_width_min_max_ex = nil;
  tiepie_hw_generator_get_width : Ttiepie_hw_generator_get_width = nil;
  tiepie_hw_generator_set_width : Ttiepie_hw_generator_set_width = nil;
  tiepie_hw_generator_verify_width : Ttiepie_hw_generator_verify_width = nil;
  tiepie_hw_generator_verify_width_ex : Ttiepie_hw_generator_verify_width_ex = nil;
  tiepie_hw_generator_has_edge_time : Ttiepie_hw_generator_has_edge_time = nil;
  tiepie_hw_generator_has_edge_time_ex : Ttiepie_hw_generator_has_edge_time_ex = nil;
  tiepie_hw_generator_get_leading_edge_time_min : Ttiepie_hw_generator_get_leading_edge_time_min = nil;
  tiepie_hw_generator_get_leading_edge_time_max : Ttiepie_hw_generator_get_leading_edge_time_max = nil;
  tiepie_hw_generator_get_leading_edge_time_min_max_ex : Ttiepie_hw_generator_get_leading_edge_time_min_max_ex = nil;
  tiepie_hw_generator_get_leading_edge_time : Ttiepie_hw_generator_get_leading_edge_time = nil;
  tiepie_hw_generator_set_leading_edge_time : Ttiepie_hw_generator_set_leading_edge_time = nil;
  tiepie_hw_generator_verify_leading_edge_time : Ttiepie_hw_generator_verify_leading_edge_time = nil;
  tiepie_hw_generator_verify_leading_edge_time_ex : Ttiepie_hw_generator_verify_leading_edge_time_ex = nil;
  tiepie_hw_generator_get_trailing_edge_time_min : Ttiepie_hw_generator_get_trailing_edge_time_min = nil;
  tiepie_hw_generator_get_trailing_edge_time_max : Ttiepie_hw_generator_get_trailing_edge_time_max = nil;
  tiepie_hw_generator_get_trailing_edge_time_min_max_ex : Ttiepie_hw_generator_get_trailing_edge_time_min_max_ex = nil;
  tiepie_hw_generator_get_trailing_edge_time : Ttiepie_hw_generator_get_trailing_edge_time = nil;
  tiepie_hw_generator_set_trailing_edge_time : Ttiepie_hw_generator_set_trailing_edge_time = nil;
  tiepie_hw_generator_verify_trailing_edge_time : Ttiepie_hw_generator_verify_trailing_edge_time = nil;
  tiepie_hw_generator_verify_trailing_edge_time_ex : Ttiepie_hw_generator_verify_trailing_edge_time_ex = nil;
  tiepie_hw_generator_has_data : Ttiepie_hw_generator_has_data = nil;
  tiepie_hw_generator_has_data_ex : Ttiepie_hw_generator_has_data_ex = nil;
  tiepie_hw_generator_get_data_length_min : Ttiepie_hw_generator_get_data_length_min = nil;
  tiepie_hw_generator_get_data_length_max : Ttiepie_hw_generator_get_data_length_max = nil;
  tiepie_hw_generator_get_data_length_min_max_ex : Ttiepie_hw_generator_get_data_length_min_max_ex = nil;
  tiepie_hw_generator_get_data_length : Ttiepie_hw_generator_get_data_length = nil;
  tiepie_hw_generator_verify_data_length : Ttiepie_hw_generator_verify_data_length = nil;
  tiepie_hw_generator_verify_data_length_ex : Ttiepie_hw_generator_verify_data_length_ex = nil;
  tiepie_hw_generator_set_data : Ttiepie_hw_generator_set_data = nil;
  tiepie_hw_generator_set_data_ex : Ttiepie_hw_generator_set_data_ex = nil;
  tiepie_hw_generator_get_data_raw_type : Ttiepie_hw_generator_get_data_raw_type = nil;
  tiepie_hw_generator_get_data_raw_value_range : Ttiepie_hw_generator_get_data_raw_value_range = nil;
  tiepie_hw_generator_get_data_raw_value_min : Ttiepie_hw_generator_get_data_raw_value_min = nil;
  tiepie_hw_generator_get_data_raw_value_zero : Ttiepie_hw_generator_get_data_raw_value_zero = nil;
  tiepie_hw_generator_get_data_raw_value_max : Ttiepie_hw_generator_get_data_raw_value_max = nil;
  tiepie_hw_generator_set_data_raw : Ttiepie_hw_generator_set_data_raw = nil;
  tiepie_hw_generator_set_data_raw_ex : Ttiepie_hw_generator_set_data_raw_ex = nil;
  tiepie_hw_generator_get_modes : Ttiepie_hw_generator_get_modes = nil;
  tiepie_hw_generator_get_modes_ex : Ttiepie_hw_generator_get_modes_ex = nil;
  tiepie_hw_generator_get_modes_native : Ttiepie_hw_generator_get_modes_native = nil;
  tiepie_hw_generator_get_mode : Ttiepie_hw_generator_get_mode = nil;
  tiepie_hw_generator_set_mode : Ttiepie_hw_generator_set_mode = nil;
  tiepie_hw_generator_is_burst_active : Ttiepie_hw_generator_is_burst_active = nil;
  tiepie_hw_generator_get_burst_count_min : Ttiepie_hw_generator_get_burst_count_min = nil;
  tiepie_hw_generator_get_burst_count_max : Ttiepie_hw_generator_get_burst_count_max = nil;
  tiepie_hw_generator_get_burst_count_min_max_ex : Ttiepie_hw_generator_get_burst_count_min_max_ex = nil;
  tiepie_hw_generator_get_burst_count : Ttiepie_hw_generator_get_burst_count = nil;
  tiepie_hw_generator_set_burst_count : Ttiepie_hw_generator_set_burst_count = nil;
  tiepie_hw_generator_get_burst_sample_count_min : Ttiepie_hw_generator_get_burst_sample_count_min = nil;
  tiepie_hw_generator_get_burst_sample_count_max : Ttiepie_hw_generator_get_burst_sample_count_max = nil;
  tiepie_hw_generator_get_burst_sample_count_min_max_ex : Ttiepie_hw_generator_get_burst_sample_count_min_max_ex = nil;
  tiepie_hw_generator_get_burst_sample_count : Ttiepie_hw_generator_get_burst_sample_count = nil;
  tiepie_hw_generator_set_burst_sample_count : Ttiepie_hw_generator_set_burst_sample_count = nil;
  tiepie_hw_generator_get_burst_segment_count_min : Ttiepie_hw_generator_get_burst_segment_count_min = nil;
  tiepie_hw_generator_get_burst_segment_count_max : Ttiepie_hw_generator_get_burst_segment_count_max = nil;
  tiepie_hw_generator_get_burst_segment_count_min_max_ex : Ttiepie_hw_generator_get_burst_segment_count_min_max_ex = nil;
  tiepie_hw_generator_get_burst_segment_count : Ttiepie_hw_generator_get_burst_segment_count = nil;
  tiepie_hw_generator_set_burst_segment_count : Ttiepie_hw_generator_set_burst_segment_count = nil;
  tiepie_hw_generator_verify_burst_segment_count : Ttiepie_hw_generator_verify_burst_segment_count = nil;
  tiepie_hw_generator_verify_burst_segment_count_ex : Ttiepie_hw_generator_verify_burst_segment_count_ex = nil;
  tiepie_hw_server_connect : Ttiepie_hw_server_connect = nil;
  tiepie_hw_server_disconnect : Ttiepie_hw_server_disconnect = nil;
  tiepie_hw_server_remove : Ttiepie_hw_server_remove = nil;
  tiepie_hw_server_get_status : Ttiepie_hw_server_get_status = nil;
  tiepie_hw_server_get_last_error : Ttiepie_hw_server_get_last_error = nil;
  tiepie_hw_server_get_url : Ttiepie_hw_server_get_url = nil;
  tiepie_hw_server_get_id : Ttiepie_hw_server_get_id = nil;
  tiepie_hw_server_get_ip_address : Ttiepie_hw_server_get_ip_address = nil;
  tiepie_hw_server_get_ip_port : Ttiepie_hw_server_get_ip_port = nil;
  tiepie_hw_server_get_name : Ttiepie_hw_server_get_name = nil;
  tiepie_hw_server_get_description : Ttiepie_hw_server_get_description = nil;
  tiepie_hw_server_get_version : Ttiepie_hw_server_get_version = nil;
  tiepie_hw_pointerarray_new : Ttiepie_hw_pointerarray_new = nil;
  tiepie_hw_pointerarray_set : Ttiepie_hw_pointerarray_set = nil;
  tiepie_hw_pointerarray_delete : Ttiepie_hw_pointerarray_delete = nil;

function tiepie_hw_load(const ALibTiePieFileName : string = TIEPIE_HW_FILENAME_DEFAULT) : Boolean;
function tiepie_hw_unload : Boolean;

{$else} // LIBTIEPIE_STATIC

procedure tiepie_hw_init; cdecl;
function tiepie_hw_is_initialized : ByteBool; cdecl;
procedure tiepie_hw_fini; cdecl;
function tiepie_hw_get_version : Ptiepie_hw_version; cdecl;
function tiepie_hw_get_config(buffer : PByte; length : LongWord) : LongWord; cdecl;
function tiepie_hw_get_last_status : Ttiepie_hw_status; cdecl;
function tiepie_hw_get_last_status_str : PAnsiChar; cdecl;
procedure tiepie_hw_devicelist_update; cdecl;
function tiepie_hw_devicelist_get_usb_hotplug_detect_enabled : ByteBool; cdecl;
function tiepie_hw_devicelist_set_usb_hotplug_detect_enabled(value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_devicelist_get_count : LongWord; cdecl;
function tiepie_hw_devicelist_get_item_by_product_id(product_id : Ttiepie_hw_productid) : Ttiepie_hw_handle; cdecl;
function tiepie_hw_devicelist_get_item_by_index(index : LongWord) : Ttiepie_hw_handle; cdecl;
function tiepie_hw_devicelist_get_item_by_serial_number(serial_number : LongWord) : Ttiepie_hw_handle; cdecl;
function tiepie_hw_devicelistitem_open_device(handle : Ttiepie_hw_handle; device_type : LongWord) : Ttiepie_hw_handle; cdecl;
function tiepie_hw_devicelistitem_open_oscilloscope(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl;
function tiepie_hw_devicelistitem_open_generator(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl;
function tiepie_hw_devicelist_get_demo_device_info : Ptiepie_hw_demo_info; cdecl;
function tiepie_hw_devicelist_create_demo_device(product_id : Ttiepie_hw_productid) : LongWord; cdecl;
function tiepie_hw_devicelist_create_combined_device(handles : Ptiepie_hw_handle; count : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelist_create_and_open_combined_device(handles : Ptiepie_hw_handle; count : LongWord) : Ttiepie_hw_handle; cdecl;
procedure tiepie_hw_devicelist_remove_device(serial_number : LongWord; force : ByteBool); cdecl;
procedure tiepie_hw_devicelist_remove_unused_devices; cdecl;
function tiepie_hw_devicelistitem_is_demo(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_devicelistitem_can_open(handle : Ttiepie_hw_handle; device_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_devicelistitem_opened_by(handle : Ttiepie_hw_handle; device_type : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitem_get_product_id(handle : Ttiepie_hw_handle) : Ttiepie_hw_productid; cdecl;
function tiepie_hw_devicelistitem_get_name(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitem_get_name_short(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitem_get_name_shortest(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitem_get_calibration_date(handle : Ttiepie_hw_handle) : Ttiepie_hw_date; cdecl;
function tiepie_hw_devicelistitem_get_serial_number(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_devicelistitem_get_ip_address(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitem_get_ip_port(handle : Ttiepie_hw_handle) : Word; cdecl;
function tiepie_hw_devicelistitem_has_server(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_devicelistitem_get_server(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl;
function tiepie_hw_devicelistitem_get_types(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_devicelistitem_get_contained_serial_numbers(handle : Ttiepie_hw_handle; buffer : PLongWord; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitemcombined_get_product_id(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Ttiepie_hw_productid; cdecl;
function tiepie_hw_devicelistitemcombined_get_name(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitemcombined_get_name_short(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitemcombined_get_name_shortest(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_devicelistitemcombined_get_calibration_date(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Ttiepie_hw_date; cdecl;
function tiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Word; cdecl;
procedure tiepie_hw_devicelist_set_callback_device_added(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl;
procedure tiepie_hw_devicelist_set_callback_device_removed(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl;
procedure tiepie_hw_devicelist_set_callback_device_can_open_changed(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl;
function tiepie_hw_network_get_auto_detect_enabled : ByteBool; cdecl;
function tiepie_hw_network_set_auto_detect_enabled(value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_network_servers_add(url : PAnsiChar; length : LongWord; handle : Ptiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_network_servers_remove(url : PAnsiChar; length : LongWord; force : ByteBool) : ByteBool; cdecl;
function tiepie_hw_network_servers_get_count : LongWord; cdecl;
function tiepie_hw_network_servers_get_by_index(index : LongWord) : Ttiepie_hw_handle; cdecl;
function tiepie_hw_network_servers_get_by_url(url : PAnsiChar; length : LongWord) : Ttiepie_hw_handle; cdecl;
procedure tiepie_hw_network_servers_set_callback_added(callback : Ttiepie_hw_handle_callback; data : Pointer); cdecl;
procedure tiepie_hw_object_close(handle : Ttiepie_hw_handle); cdecl;
function tiepie_hw_object_is_removed(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_object_get_interfaces(handle : Ttiepie_hw_handle) : UInt64; cdecl;
procedure tiepie_hw_object_set_event_callback(handle : Ttiepie_hw_handle; callback : Ttiepie_hw_event_callback; data : Pointer); cdecl;
function tiepie_hw_device_get_calibration_date(handle : Ttiepie_hw_handle) : Ttiepie_hw_date; cdecl;
function tiepie_hw_device_get_serial_number(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_device_get_ip_address(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_device_get_ip_port(handle : Ttiepie_hw_handle) : Word; cdecl;
function tiepie_hw_device_get_product_id(handle : Ttiepie_hw_handle) : Ttiepie_hw_productid; cdecl;
function tiepie_hw_device_get_type(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_device_get_name(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_device_get_name_short(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_device_get_name_shortest(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_device_has_wireless_trigger_module(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_device_has_battery(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_device_has_battery_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl;
function tiepie_hw_device_get_battery_charge(handle : Ttiepie_hw_handle) : ShortInt; cdecl;
function tiepie_hw_device_get_battery_charge_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ShortInt; cdecl;
function tiepie_hw_device_get_battery_time_to_empty(handle : Ttiepie_hw_handle) : LongInt; cdecl;
function tiepie_hw_device_get_battery_time_to_empty_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : LongInt; cdecl;
function tiepie_hw_device_get_battery_time_to_full(handle : Ttiepie_hw_handle) : LongInt; cdecl;
function tiepie_hw_device_get_battery_time_to_full_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : LongInt; cdecl;
function tiepie_hw_device_is_battery_charger_connected(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_device_is_battery_charger_connected_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl;
function tiepie_hw_device_is_battery_charging(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_device_is_battery_charging_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl;
function tiepie_hw_device_is_battery_broken(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_device_is_battery_broken_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl;
function tiepie_hw_device_trigger_get_input_count(handle : Ttiepie_hw_handle) : Word; cdecl;
function tiepie_hw_device_trigger_get_input_index_by_id(handle : Ttiepie_hw_handle; id : LongWord) : Word; cdecl;
function tiepie_hw_oscilloscope_trigger_input_is_triggered(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl;
function tiepie_hw_device_trigger_input_get_enabled(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl;
function tiepie_hw_device_trigger_input_set_enabled(handle : Ttiepie_hw_handle; input : Word; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_device_trigger_input_get_kinds(handle : Ttiepie_hw_handle; input : Word) : UInt64; cdecl;
function tiepie_hw_oscilloscope_trigger_input_get_kinds_ex(handle : Ttiepie_hw_handle; input : Word; measure_mode : LongWord) : UInt64; cdecl;
function tiepie_hw_device_trigger_input_get_kind(handle : Ttiepie_hw_handle; input : Word) : UInt64; cdecl;
function tiepie_hw_device_trigger_input_set_kind(handle : Ttiepie_hw_handle; input : Word; value : UInt64) : UInt64; cdecl;
function tiepie_hw_device_trigger_input_is_available(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_trigger_input_is_available_ex(handle : Ttiepie_hw_handle; input : Word; measure_mode : LongWord) : ByteBool; cdecl;
function tiepie_hw_device_trigger_input_get_id(handle : Ttiepie_hw_handle; input : Word) : LongWord; cdecl;
function tiepie_hw_device_trigger_input_get_name(handle : Ttiepie_hw_handle; input : Word; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_device_trigger_get_output_count(handle : Ttiepie_hw_handle) : Word; cdecl;
function tiepie_hw_device_trigger_get_output_index_by_id(handle : Ttiepie_hw_handle; id : LongWord) : Word; cdecl;
function tiepie_hw_device_trigger_output_get_enabled(handle : Ttiepie_hw_handle; output : Word) : ByteBool; cdecl;
function tiepie_hw_device_trigger_output_set_enabled(handle : Ttiepie_hw_handle; output : Word; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_device_trigger_output_get_events(handle : Ttiepie_hw_handle; output : Word) : UInt64; cdecl;
function tiepie_hw_device_trigger_output_get_event(handle : Ttiepie_hw_handle; output : Word) : UInt64; cdecl;
function tiepie_hw_device_trigger_output_set_event(handle : Ttiepie_hw_handle; output : Word; value : UInt64) : UInt64; cdecl;
function tiepie_hw_device_trigger_output_get_id(handle : Ttiepie_hw_handle; output : Word) : LongWord; cdecl;
function tiepie_hw_device_trigger_output_get_name(handle : Ttiepie_hw_handle; output : Word; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_device_trigger_output_trigger(handle : Ttiepie_hw_handle; output : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_get_channel_count(handle : Ttiepie_hw_handle) : Word; cdecl;
function tiepie_hw_oscilloscope_channel_is_available(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_is_available_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; sample_rate : Double; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_get_connector_type(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_is_differential(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_is_isolated(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_get_impedance(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_get_bandwidths(handle : Ttiepie_hw_handle; ch : Word; list : PDouble; length : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_get_bandwidth(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_set_bandwidth(handle : Ttiepie_hw_handle; ch : Word; bandwidth : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_get_couplings(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl;
function tiepie_hw_oscilloscope_channel_get_coupling(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl;
function tiepie_hw_oscilloscope_channel_set_coupling(handle : Ttiepie_hw_handle; ch : Word; coupling : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_channel_get_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_set_enabled(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_get_auto_ranging(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_set_auto_ranging(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_get_ranges(handle : Ttiepie_hw_handle; ch : Word; list : PDouble; length : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_get_ranges_ex(handle : Ttiepie_hw_handle; ch : Word; coupling : UInt64; list : PDouble; length : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_get_range(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_set_range(handle : Ttiepie_hw_handle; ch : Word; range : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_has_safeground(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_get_safeground_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_set_safeground_enabled(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_get_safeground_threshold_min(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_get_safeground_threshold_max(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_get_safeground_threshold(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_set_safeground_threshold(handle : Ttiepie_hw_handle; ch : Word; threshold : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_verify_safeground_threshold(handle : Ttiepie_hw_handle; ch : Word; threshold : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_has_trigger(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_has_trigger_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_is_available(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_is_available_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; sample_rate : Double; resolution : Byte; channel_enabled, channel_trigger_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_is_triggered(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_set_enabled(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_kinds(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_kinds_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord) : UInt64; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_kind(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_set_kind(handle : Ttiepie_hw_handle; ch : Word; value : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_level_modes(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_level_mode(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_set_level_mode(handle : Ttiepie_hw_handle; ch : Word; value : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_level_count(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_level(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_set_level(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_hysteresis(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_set_hysteresis(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_conditions(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_conditions_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; trigger_kind : UInt64) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_condition(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_set_condition(handle : Ttiepie_hw_handle; ch : Word; value : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_time_count(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_get_time(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_set_time(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_verify_time(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_trigger_verify_time_ex(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double; measure_mode : LongWord; sample_rate : Double; trigger_kind : UInt64; trigger_condition : LongWord) : Double; cdecl;
function tiepie_hw_oscilloscope_is_demo(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_demo_get_signal(handle : Ttiepie_hw_handle; ch : Word) : Ttiepie_hw_demosignal; cdecl;
function tiepie_hw_oscilloscope_channel_demo_set_signal(handle : Ttiepie_hw_handle; ch : Word; value : Ttiepie_hw_demosignal) : Ttiepie_hw_demosignal; cdecl;
function tiepie_hw_oscilloscope_channel_demo_has_amplitude(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_demo_get_amplitude(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_set_amplitude(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_has_frequency(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_demo_get_frequency(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_set_frequency(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_has_offset(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_demo_get_offset(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_set_offset(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_has_symmetry(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_demo_get_symmetry(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_set_symmetry(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_has_phase(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_demo_get_phase(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_set_phase(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_demo_has_noise_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_demo_get_noise_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_demo_set_noise_enabled(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_get_data(handle : Ttiepie_hw_handle; buffers : PPSingle; channel_count : Word; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_1ch(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_2ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_3ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_4ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_5ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_6ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_7ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6, buffer_ch7 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_8ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6, buffer_ch7, buffer_ch8 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_valid_pre_sample_count(handle : Ttiepie_hw_handle) : UInt64; cdecl;
procedure tiepie_hw_oscilloscope_channel_get_data_value_range(handle : Ttiepie_hw_handle; ch : Word; min, max : PDouble); cdecl;
function tiepie_hw_oscilloscope_channel_get_data_value_min(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_channel_get_data_value_max(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl;
function tiepie_hw_oscilloscope_get_data_raw(handle : Ttiepie_hw_handle; buffers : PPointer; channel_count : Word; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_raw_1ch(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_raw_2ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_raw_3ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_raw_4ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_raw_5ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_raw_6ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_raw_7ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6, buffer_ch7 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_data_raw_8ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6, buffer_ch7, buffer_ch8 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_channel_get_data_raw_type(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl;
procedure tiepie_hw_oscilloscope_channel_get_data_raw_value_range(handle : Ttiepie_hw_handle; ch : Word; min, zero, max : PInt64); cdecl;
function tiepie_hw_oscilloscope_channel_get_data_raw_value_min(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl;
function tiepie_hw_oscilloscope_channel_get_data_raw_value_zero(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl;
function tiepie_hw_oscilloscope_channel_get_data_raw_value_max(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl;
function tiepie_hw_oscilloscope_channel_is_range_max_reachable(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_start(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_stop(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_force_trigger(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_get_measure_modes(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_measure_mode(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_set_measure_mode(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_is_running(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_is_triggered(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_is_timeout_triggered(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_is_force_triggered(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_is_data_ready(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_is_data_overflow(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_get_auto_resolution_modes(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_auto_resolution_mode(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_set_auto_resolution_mode(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_resolutions(handle : Ttiepie_hw_handle; list : PByte; length : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_resolution(handle : Ttiepie_hw_handle) : Byte; cdecl;
function tiepie_hw_oscilloscope_set_resolution(handle : Ttiepie_hw_handle; value : Byte) : Byte; cdecl;
function tiepie_hw_oscilloscope_is_resolution_enhanced(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_is_resolution_enhanced_ex(handle : Ttiepie_hw_handle; value : Byte) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_get_clock_sources(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_clock_source(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_set_clock_source(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_clock_source_frequencies(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_clock_source_frequencies_ex(handle : Ttiepie_hw_handle; value : LongWord; list : PDouble; length : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_clock_source_frequency(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_oscilloscope_set_clock_source_frequency(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_get_clock_outputs(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_clock_output(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_set_clock_output(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_clock_output_frequencies(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_clock_output_frequencies_ex(handle : Ttiepie_hw_handle; clock_output : LongWord; list : PDouble; length : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_clock_output_frequency(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_oscilloscope_set_clock_output_frequency(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_get_sample_rate_max(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_oscilloscope_get_sample_rate(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_oscilloscope_set_sample_rate(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_verify_sample_rate(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_verify_sample_rate_ex(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : Double; cdecl;
procedure tiepie_hw_oscilloscope_verify_sample_rates_ex(handle : Ttiepie_hw_handle; values : PDouble; count, measure_mode, auto_resolution_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word); cdecl;
function tiepie_hw_oscilloscope_get_record_length_max(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_record_length_max_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord; resolution : Byte) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_record_length(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_oscilloscope_set_record_length(handle : Ttiepie_hw_handle; record_length : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_verify_record_length(handle : Ttiepie_hw_handle; record_length : UInt64) : UInt64; cdecl;
function tiepie_hw_oscilloscope_verify_record_length_ex(handle : Ttiepie_hw_handle; record_length : UInt64; measure_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : UInt64; cdecl;
function tiepie_hw_oscilloscope_get_pre_sample_ratio(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_oscilloscope_set_pre_sample_ratio(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_get_segment_count_max(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_segment_count_max_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_get_segment_count(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_oscilloscope_set_segment_count(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_verify_segment_count(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
function tiepie_hw_oscilloscope_verify_segment_count_ex(handle : Ttiepie_hw_handle; value, measure_mode : LongWord; record_length : UInt64; channel_enabled : PByteBool; channel_count : Word) : LongWord; cdecl;
function tiepie_hw_oscilloscope_has_trigger(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_has_trigger_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_trigger_get_timeout(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_set_timeout(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_verify_timeout(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_verify_timeout_ex(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; sample_rate : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_has_delay(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_trigger_has_delay_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_trigger_get_delay_max(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_get_delay_max_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord; sample_rate : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_get_delay(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_set_delay(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_verify_delay(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_trigger_verify_delay_ex(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; sample_rate : Double) : Double; cdecl;
function tiepie_hw_oscilloscope_has_presamples_valid(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_has_presamples_valid_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_get_presamples_valid(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_set_presamples_valid(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_has_sureconnect(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_channel_has_sureconnect(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_start_sureconnect(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_start_sureconnect_ex(handle : Ttiepie_hw_handle; channel_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_is_sureconnect_completed(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_oscilloscope_get_sureconnect_data(handle : Ttiepie_hw_handle; buffer : Ptiepie_hw_tristate; channel_count : Word) : Word; cdecl;
function tiepie_hw_generator_get_connector_type(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_generator_is_differential(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_get_impedance(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_resolution(handle : Ttiepie_hw_handle) : Byte; cdecl;
function tiepie_hw_generator_get_output_value_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_output_value_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_output_value_min_max(handle : Ttiepie_hw_handle; min, max : PDouble); cdecl;
function tiepie_hw_generator_is_controllable(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_is_running(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_get_status(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_generator_get_output_enable(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_set_output_enable(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_generator_has_output_invert(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_get_output_invert(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_set_output_invert(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_generator_start(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_stop(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_get_signal_types(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_generator_get_signal_type(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_generator_set_signal_type(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
function tiepie_hw_generator_has_amplitude(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_has_amplitude_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_generator_get_amplitude_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_amplitude_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_amplitude_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PDouble); cdecl;
function tiepie_hw_generator_get_amplitude(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_amplitude(handle : Ttiepie_hw_handle; amplitude : Double) : Double; cdecl;
function tiepie_hw_generator_verify_amplitude(handle : Ttiepie_hw_handle; amplitude : Double) : Double; cdecl;
function tiepie_hw_generator_verify_amplitude_ex(handle : Ttiepie_hw_handle; amplitude : Double; signal_type, amplitude_range_index : LongWord; offset : Double; output_invert : ByteBool) : Double; cdecl;
function tiepie_hw_generator_get_amplitude_ranges(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl;
function tiepie_hw_generator_get_amplitude_range(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_amplitude_range(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_get_amplitude_auto_ranging(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_set_amplitude_auto_ranging(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl;
function tiepie_hw_generator_has_offset(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_has_offset_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_generator_get_offset_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_offset_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_offset_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PDouble); cdecl;
function tiepie_hw_generator_get_offset(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_offset(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_offset(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_offset_ex(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord; amplitude : Double; output_invert : ByteBool) : Double; cdecl;
function tiepie_hw_generator_get_frequency_modes(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_generator_get_frequency_modes_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : LongWord; cdecl;
function tiepie_hw_generator_get_frequency_mode(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_generator_set_frequency_mode(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl;
function tiepie_hw_generator_has_frequency(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_has_frequency_ex(handle : Ttiepie_hw_handle; frequency_mode, signal_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_generator_get_frequency_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_frequency_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_frequency_min_max(handle : Ttiepie_hw_handle; frequency_mode : LongWord; min, max : PDouble); cdecl;
procedure tiepie_hw_generator_get_frequency_min_max_ex(handle : Ttiepie_hw_handle; frequency_mode, signal_type : LongWord; min, max : PDouble); cdecl;
function tiepie_hw_generator_get_frequency(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_frequency(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_frequency(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_frequency_ex(handle : Ttiepie_hw_handle; value : Double; frequency_mode, signal_type : LongWord; data_length : UInt64; width : Double) : Double; cdecl;
function tiepie_hw_generator_has_phase(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_has_phase_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_generator_get_phase_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_phase_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_phase_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PDouble); cdecl;
function tiepie_hw_generator_get_phase(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_phase(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_phase(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_phase_ex(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord) : Double; cdecl;
function tiepie_hw_generator_has_symmetry(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_has_symmetry_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_generator_get_symmetry_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_symmetry_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_symmetry_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PDouble); cdecl;
function tiepie_hw_generator_get_symmetry(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_symmetry(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_symmetry(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_symmetry_ex(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord) : Double; cdecl;
function tiepie_hw_generator_has_width(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_has_width_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_generator_get_width_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_width_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_width_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency : Double; min, max : PDouble); cdecl;
function tiepie_hw_generator_get_width(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_width(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_width(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_width_ex(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord; signal_frequency : Double) : Double; cdecl;
function tiepie_hw_generator_has_edge_time(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_has_edge_time_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_generator_get_leading_edge_time_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_leading_edge_time_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_leading_edge_time_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency, symmetry, width, trailing_edge_time : Double; min, max : PDouble); cdecl;
function tiepie_hw_generator_get_leading_edge_time(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_leading_edge_time(handle : Ttiepie_hw_handle; leading_edge_time : Double) : Double; cdecl;
function tiepie_hw_generator_verify_leading_edge_time(handle : Ttiepie_hw_handle; leading_edge_time : Double) : Double; cdecl;
function tiepie_hw_generator_verify_leading_edge_time_ex(handle : Ttiepie_hw_handle; leading_edge_time : Double; signal_type : LongWord; signal_frequency, symmetry, width, trailing_edge_time : Double) : Double; cdecl;
function tiepie_hw_generator_get_trailing_edge_time_min(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_get_trailing_edge_time_max(handle : Ttiepie_hw_handle) : Double; cdecl;
procedure tiepie_hw_generator_get_trailing_edge_time_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency, symmetry, width, leading_edge_time : Double; min, max : PDouble); cdecl;
function tiepie_hw_generator_get_trailing_edge_time(handle : Ttiepie_hw_handle) : Double; cdecl;
function tiepie_hw_generator_set_trailing_edge_time(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_trailing_edge_time(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl;
function tiepie_hw_generator_verify_trailing_edge_time_ex(handle : Ttiepie_hw_handle; trailing_edge_time : Double; signal_type : LongWord; signal_frequency, symmetry, width, leading_edge_time : Double) : Double; cdecl;
function tiepie_hw_generator_has_data(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_has_data_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl;
function tiepie_hw_generator_get_data_length_min(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_get_data_length_max(handle : Ttiepie_hw_handle) : UInt64; cdecl;
procedure tiepie_hw_generator_get_data_length_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PUInt64); cdecl;
function tiepie_hw_generator_get_data_length(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_verify_data_length(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
function tiepie_hw_generator_verify_data_length_ex(handle : Ttiepie_hw_handle; value : UInt64; signal_type : LongWord) : UInt64; cdecl;
procedure tiepie_hw_generator_set_data(handle : Ttiepie_hw_handle; buffer : PSingle; sample_count : UInt64); cdecl;
procedure tiepie_hw_generator_set_data_ex(handle : Ttiepie_hw_handle; buffer : PSingle; sample_count : UInt64; signal_type : LongWord); cdecl;
function tiepie_hw_generator_get_data_raw_type(handle : Ttiepie_hw_handle) : LongWord; cdecl;
procedure tiepie_hw_generator_get_data_raw_value_range(handle : Ttiepie_hw_handle; min, zero, max : PInt64); cdecl;
function tiepie_hw_generator_get_data_raw_value_min(handle : Ttiepie_hw_handle) : Int64; cdecl;
function tiepie_hw_generator_get_data_raw_value_zero(handle : Ttiepie_hw_handle) : Int64; cdecl;
function tiepie_hw_generator_get_data_raw_value_max(handle : Ttiepie_hw_handle) : Int64; cdecl;
procedure tiepie_hw_generator_set_data_raw(handle : Ttiepie_hw_handle; buffer : Pointer; sample_count : UInt64); cdecl;
procedure tiepie_hw_generator_set_data_raw_ex(handle : Ttiepie_hw_handle; buffer : Pointer; sample_count : UInt64; signal_type : LongWord); cdecl;
function tiepie_hw_generator_get_modes(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_get_modes_ex(handle : Ttiepie_hw_handle; signal_type, frequency_mode : LongWord) : UInt64; cdecl;
function tiepie_hw_generator_get_modes_native(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_get_mode(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_set_mode(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
function tiepie_hw_generator_is_burst_active(handle : Ttiepie_hw_handle) : ByteBool; cdecl;
function tiepie_hw_generator_get_burst_count_min(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_get_burst_count_max(handle : Ttiepie_hw_handle) : UInt64; cdecl;
procedure tiepie_hw_generator_get_burst_count_min_max_ex(handle : Ttiepie_hw_handle; generator_mode : UInt64; min, max : PUInt64); cdecl;
function tiepie_hw_generator_get_burst_count(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_set_burst_count(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
function tiepie_hw_generator_get_burst_sample_count_min(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_get_burst_sample_count_max(handle : Ttiepie_hw_handle) : UInt64; cdecl;
procedure tiepie_hw_generator_get_burst_sample_count_min_max_ex(handle : Ttiepie_hw_handle; generator_mode : UInt64; min, max : PUInt64); cdecl;
function tiepie_hw_generator_get_burst_sample_count(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_set_burst_sample_count(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
function tiepie_hw_generator_get_burst_segment_count_min(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_get_burst_segment_count_max(handle : Ttiepie_hw_handle) : UInt64; cdecl;
procedure tiepie_hw_generator_get_burst_segment_count_min_max_ex(handle : Ttiepie_hw_handle; generator_mode : UInt64; signal_type, frequency_mode : LongWord; frequency : Double; data_length : UInt64; min, max : PUInt64); cdecl;
function tiepie_hw_generator_get_burst_segment_count(handle : Ttiepie_hw_handle) : UInt64; cdecl;
function tiepie_hw_generator_set_burst_segment_count(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
function tiepie_hw_generator_verify_burst_segment_count(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl;
function tiepie_hw_generator_verify_burst_segment_count_ex(handle : Ttiepie_hw_handle; value, generator_mode : UInt64; signal_type, frequency_mode : LongWord; frequency : Double; data_length : UInt64) : UInt64; cdecl;
function tiepie_hw_server_connect(handle : Ttiepie_hw_handle; asynchronous : ByteBool) : ByteBool; cdecl;
function tiepie_hw_server_disconnect(handle : Ttiepie_hw_handle; force : ByteBool) : ByteBool; cdecl;
function tiepie_hw_server_remove(handle : Ttiepie_hw_handle; force : ByteBool) : ByteBool; cdecl;
function tiepie_hw_server_get_status(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_server_get_last_error(handle : Ttiepie_hw_handle) : LongWord; cdecl;
function tiepie_hw_server_get_url(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_server_get_id(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_server_get_ip_address(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_server_get_ip_port(handle : Ttiepie_hw_handle) : Word; cdecl;
function tiepie_hw_server_get_name(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_server_get_description(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_server_get_version(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl;
function tiepie_hw_pointerarray_new(length : LongWord) : Ttiepie_hw_pointerarray; cdecl;
procedure tiepie_hw_pointerarray_set(ptr : Ttiepie_hw_pointerarray; index : LongWord; value : Pointer); cdecl;
procedure tiepie_hw_pointerarray_delete(ptr : Ttiepie_hw_pointerarray); cdecl;

{$endif}

// Extra routines:

function tiepie_hw_get_config_dynarray : TByteDynArray;
function tiepie_hw_get_last_status_string() : string;
function tiepie_hw_devicelistitem_opened_by_string(handle : Ttiepie_hw_handle; device_type : LongWord) : string;
function tiepie_hw_devicelistitem_get_name_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_devicelistitem_get_name_short_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_devicelistitem_get_name_shortest_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_devicelistitem_get_ip_address_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_devicelistitem_get_contained_serial_numbers_dynarray(handle : Ttiepie_hw_handle) : TLongWordDynArray;
function tiepie_hw_devicelistitemcombined_get_name_string(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : string;
function tiepie_hw_devicelistitemcombined_get_name_short_string(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : string;
function tiepie_hw_devicelistitemcombined_get_name_shortest_string(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : string;
function tiepie_hw_device_get_ip_address_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_device_get_name_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_device_get_name_short_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_device_get_name_shortest_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_device_trigger_input_get_name_string(handle : Ttiepie_hw_handle; input : Word) : string;
function tiepie_hw_device_trigger_output_get_name_string(handle : Ttiepie_hw_handle; output : Word) : string;
function tiepie_hw_oscilloscope_channel_get_bandwidths_dynarray(handle : Ttiepie_hw_handle; ch : Word) : TDoubleDynArray;
function tiepie_hw_oscilloscope_channel_get_ranges_dynarray(handle : Ttiepie_hw_handle; ch : Word) : TDoubleDynArray;
function tiepie_hw_oscilloscope_channel_get_ranges_ex_dynarray(handle : Ttiepie_hw_handle; ch : Word; coupling : UInt64) : TDoubleDynArray;
function tiepie_hw_oscilloscope_get_resolutions_dynarray(handle : Ttiepie_hw_handle) : TByteDynArray;
function tiepie_hw_oscilloscope_get_clock_source_frequencies_dynarray(handle : Ttiepie_hw_handle) : TDoubleDynArray;
function tiepie_hw_oscilloscope_get_clock_source_frequencies_ex_dynarray(handle : Ttiepie_hw_handle; value : LongWord) : TDoubleDynArray;
function tiepie_hw_oscilloscope_get_clock_output_frequencies_dynarray(handle : Ttiepie_hw_handle) : TDoubleDynArray;
function tiepie_hw_oscilloscope_get_clock_output_frequencies_ex_dynarray(handle : Ttiepie_hw_handle; clock_output : LongWord) : TDoubleDynArray;
function tiepie_hw_generator_get_amplitude_ranges_dynarray(handle : Ttiepie_hw_handle) : TDoubleDynArray;
function tiepie_hw_server_get_url_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_server_get_id_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_server_get_ip_address_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_server_get_name_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_server_get_description_string(handle : Ttiepie_hw_handle) : string;
function tiepie_hw_server_get_version_string(handle : Ttiepie_hw_handle) : string;

{$ifdef LIBTIEPIE_HW_DYNAMIC}
var
  {$ifdef MSWINDOWS}
  hLibTiePie : THandle = 0;
  {$else}
  hLibTiePie : TLibHandle = 0;
  {$endif}
{$endif LIBTIEPIE_HW_DYNAMIC}

implementation

uses
  {$if defined(LIBTIEPIE_HW_DYNAMIC) and not defined(MSWINDOWS)}
  dynlibs,
  {$ifend}
  SysUtils;

{$ifdef LIBTIEPIE_HW_DYNAMIC}

function tiepie_hw_load(const ALibTiePieFileName : string = TIEPIE_HW_FILENAME_DEFAULT) : Boolean;
begin
  if hLibTiePie = 0 then begin
    hLibTiePie := LoadLibrary(PAnsiChar(ALibTiePieFileName));

    if hLibTiePie <> 0 then begin
      tiepie_hw_init := Ttiepie_hw_init( GetProcAddress( hLibTiePie , 'tiepie_hw_init' ) );
      tiepie_hw_is_initialized := Ttiepie_hw_is_initialized( GetProcAddress( hLibTiePie , 'tiepie_hw_is_initialized' ) );
      tiepie_hw_fini := Ttiepie_hw_fini( GetProcAddress( hLibTiePie , 'tiepie_hw_fini' ) );
      tiepie_hw_get_version := Ttiepie_hw_get_version( GetProcAddress( hLibTiePie , 'tiepie_hw_get_version' ) );
      tiepie_hw_get_config := Ttiepie_hw_get_config( GetProcAddress( hLibTiePie , 'tiepie_hw_get_config' ) );
      tiepie_hw_get_last_status := Ttiepie_hw_get_last_status( GetProcAddress( hLibTiePie , 'tiepie_hw_get_last_status' ) );
      tiepie_hw_get_last_status_str := Ttiepie_hw_get_last_status_str( GetProcAddress( hLibTiePie , 'tiepie_hw_get_last_status_str' ) );
      tiepie_hw_devicelist_update := Ttiepie_hw_devicelist_update( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_update' ) );
      tiepie_hw_devicelist_get_usb_hotplug_detect_enabled := Ttiepie_hw_devicelist_get_usb_hotplug_detect_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_get_usb_hotplug_detect_enabled' ) );
      tiepie_hw_devicelist_set_usb_hotplug_detect_enabled := Ttiepie_hw_devicelist_set_usb_hotplug_detect_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_set_usb_hotplug_detect_enabled' ) );
      tiepie_hw_devicelist_get_count := Ttiepie_hw_devicelist_get_count( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_get_count' ) );
      tiepie_hw_devicelist_get_item_by_product_id := Ttiepie_hw_devicelist_get_item_by_product_id( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_get_item_by_product_id' ) );
      tiepie_hw_devicelist_get_item_by_index := Ttiepie_hw_devicelist_get_item_by_index( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_get_item_by_index' ) );
      tiepie_hw_devicelist_get_item_by_serial_number := Ttiepie_hw_devicelist_get_item_by_serial_number( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_get_item_by_serial_number' ) );
      tiepie_hw_devicelistitem_open_device := Ttiepie_hw_devicelistitem_open_device( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_open_device' ) );
      tiepie_hw_devicelistitem_open_oscilloscope := Ttiepie_hw_devicelistitem_open_oscilloscope( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_open_oscilloscope' ) );
      tiepie_hw_devicelistitem_open_generator := Ttiepie_hw_devicelistitem_open_generator( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_open_generator' ) );
      tiepie_hw_devicelist_get_demo_device_info := Ttiepie_hw_devicelist_get_demo_device_info( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_get_demo_device_info' ) );
      tiepie_hw_devicelist_create_demo_device := Ttiepie_hw_devicelist_create_demo_device( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_create_demo_device' ) );
      tiepie_hw_devicelist_create_combined_device := Ttiepie_hw_devicelist_create_combined_device( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_create_combined_device' ) );
      tiepie_hw_devicelist_create_and_open_combined_device := Ttiepie_hw_devicelist_create_and_open_combined_device( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_create_and_open_combined_device' ) );
      tiepie_hw_devicelist_remove_device := Ttiepie_hw_devicelist_remove_device( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_remove_device' ) );
      tiepie_hw_devicelist_remove_unused_devices := Ttiepie_hw_devicelist_remove_unused_devices( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_remove_unused_devices' ) );
      tiepie_hw_devicelistitem_is_demo := Ttiepie_hw_devicelistitem_is_demo( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_is_demo' ) );
      tiepie_hw_devicelistitem_can_open := Ttiepie_hw_devicelistitem_can_open( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_can_open' ) );
      tiepie_hw_devicelistitem_opened_by := Ttiepie_hw_devicelistitem_opened_by( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_opened_by' ) );
      tiepie_hw_devicelistitem_get_product_id := Ttiepie_hw_devicelistitem_get_product_id( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_product_id' ) );
      tiepie_hw_devicelistitem_get_name := Ttiepie_hw_devicelistitem_get_name( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_name' ) );
      tiepie_hw_devicelistitem_get_name_short := Ttiepie_hw_devicelistitem_get_name_short( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_name_short' ) );
      tiepie_hw_devicelistitem_get_name_shortest := Ttiepie_hw_devicelistitem_get_name_shortest( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_name_shortest' ) );
      tiepie_hw_devicelistitem_get_calibration_date := Ttiepie_hw_devicelistitem_get_calibration_date( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_calibration_date' ) );
      tiepie_hw_devicelistitem_get_serial_number := Ttiepie_hw_devicelistitem_get_serial_number( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_serial_number' ) );
      tiepie_hw_devicelistitem_get_ip_address := Ttiepie_hw_devicelistitem_get_ip_address( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_ip_address' ) );
      tiepie_hw_devicelistitem_get_ip_port := Ttiepie_hw_devicelistitem_get_ip_port( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_ip_port' ) );
      tiepie_hw_devicelistitem_has_server := Ttiepie_hw_devicelistitem_has_server( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_has_server' ) );
      tiepie_hw_devicelistitem_get_server := Ttiepie_hw_devicelistitem_get_server( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_server' ) );
      tiepie_hw_devicelistitem_get_types := Ttiepie_hw_devicelistitem_get_types( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_types' ) );
      tiepie_hw_devicelistitem_get_contained_serial_numbers := Ttiepie_hw_devicelistitem_get_contained_serial_numbers( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitem_get_contained_serial_numbers' ) );
      tiepie_hw_devicelistitemcombined_get_product_id := Ttiepie_hw_devicelistitemcombined_get_product_id( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitemcombined_get_product_id' ) );
      tiepie_hw_devicelistitemcombined_get_name := Ttiepie_hw_devicelistitemcombined_get_name( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitemcombined_get_name' ) );
      tiepie_hw_devicelistitemcombined_get_name_short := Ttiepie_hw_devicelistitemcombined_get_name_short( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitemcombined_get_name_short' ) );
      tiepie_hw_devicelistitemcombined_get_name_shortest := Ttiepie_hw_devicelistitemcombined_get_name_shortest( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitemcombined_get_name_shortest' ) );
      tiepie_hw_devicelistitemcombined_get_calibration_date := Ttiepie_hw_devicelistitemcombined_get_calibration_date( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitemcombined_get_calibration_date' ) );
      tiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count := Ttiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count' ) );
      tiepie_hw_devicelist_set_callback_device_added := Ttiepie_hw_devicelist_set_callback_device_added( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_set_callback_device_added' ) );
      tiepie_hw_devicelist_set_callback_device_removed := Ttiepie_hw_devicelist_set_callback_device_removed( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_set_callback_device_removed' ) );
      tiepie_hw_devicelist_set_callback_device_can_open_changed := Ttiepie_hw_devicelist_set_callback_device_can_open_changed( GetProcAddress( hLibTiePie , 'tiepie_hw_devicelist_set_callback_device_can_open_changed' ) );
      tiepie_hw_network_get_auto_detect_enabled := Ttiepie_hw_network_get_auto_detect_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_network_get_auto_detect_enabled' ) );
      tiepie_hw_network_set_auto_detect_enabled := Ttiepie_hw_network_set_auto_detect_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_network_set_auto_detect_enabled' ) );
      tiepie_hw_network_servers_add := Ttiepie_hw_network_servers_add( GetProcAddress( hLibTiePie , 'tiepie_hw_network_servers_add' ) );
      tiepie_hw_network_servers_remove := Ttiepie_hw_network_servers_remove( GetProcAddress( hLibTiePie , 'tiepie_hw_network_servers_remove' ) );
      tiepie_hw_network_servers_get_count := Ttiepie_hw_network_servers_get_count( GetProcAddress( hLibTiePie , 'tiepie_hw_network_servers_get_count' ) );
      tiepie_hw_network_servers_get_by_index := Ttiepie_hw_network_servers_get_by_index( GetProcAddress( hLibTiePie , 'tiepie_hw_network_servers_get_by_index' ) );
      tiepie_hw_network_servers_get_by_url := Ttiepie_hw_network_servers_get_by_url( GetProcAddress( hLibTiePie , 'tiepie_hw_network_servers_get_by_url' ) );
      tiepie_hw_network_servers_set_callback_added := Ttiepie_hw_network_servers_set_callback_added( GetProcAddress( hLibTiePie , 'tiepie_hw_network_servers_set_callback_added' ) );
      tiepie_hw_object_close := Ttiepie_hw_object_close( GetProcAddress( hLibTiePie , 'tiepie_hw_object_close' ) );
      tiepie_hw_object_is_removed := Ttiepie_hw_object_is_removed( GetProcAddress( hLibTiePie , 'tiepie_hw_object_is_removed' ) );
      tiepie_hw_object_get_interfaces := Ttiepie_hw_object_get_interfaces( GetProcAddress( hLibTiePie , 'tiepie_hw_object_get_interfaces' ) );
      tiepie_hw_object_set_event_callback := Ttiepie_hw_object_set_event_callback( GetProcAddress( hLibTiePie , 'tiepie_hw_object_set_event_callback' ) );
      tiepie_hw_device_get_calibration_date := Ttiepie_hw_device_get_calibration_date( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_calibration_date' ) );
      tiepie_hw_device_get_serial_number := Ttiepie_hw_device_get_serial_number( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_serial_number' ) );
      tiepie_hw_device_get_ip_address := Ttiepie_hw_device_get_ip_address( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_ip_address' ) );
      tiepie_hw_device_get_ip_port := Ttiepie_hw_device_get_ip_port( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_ip_port' ) );
      tiepie_hw_device_get_product_id := Ttiepie_hw_device_get_product_id( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_product_id' ) );
      tiepie_hw_device_get_type := Ttiepie_hw_device_get_type( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_type' ) );
      tiepie_hw_device_get_name := Ttiepie_hw_device_get_name( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_name' ) );
      tiepie_hw_device_get_name_short := Ttiepie_hw_device_get_name_short( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_name_short' ) );
      tiepie_hw_device_get_name_shortest := Ttiepie_hw_device_get_name_shortest( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_name_shortest' ) );
      tiepie_hw_device_has_wireless_trigger_module := Ttiepie_hw_device_has_wireless_trigger_module( GetProcAddress( hLibTiePie , 'tiepie_hw_device_has_wireless_trigger_module' ) );
      tiepie_hw_device_has_battery := Ttiepie_hw_device_has_battery( GetProcAddress( hLibTiePie , 'tiepie_hw_device_has_battery' ) );
      tiepie_hw_device_has_battery_ex := Ttiepie_hw_device_has_battery_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_device_has_battery_ex' ) );
      tiepie_hw_device_get_battery_charge := Ttiepie_hw_device_get_battery_charge( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_battery_charge' ) );
      tiepie_hw_device_get_battery_charge_ex := Ttiepie_hw_device_get_battery_charge_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_battery_charge_ex' ) );
      tiepie_hw_device_get_battery_time_to_empty := Ttiepie_hw_device_get_battery_time_to_empty( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_battery_time_to_empty' ) );
      tiepie_hw_device_get_battery_time_to_empty_ex := Ttiepie_hw_device_get_battery_time_to_empty_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_battery_time_to_empty_ex' ) );
      tiepie_hw_device_get_battery_time_to_full := Ttiepie_hw_device_get_battery_time_to_full( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_battery_time_to_full' ) );
      tiepie_hw_device_get_battery_time_to_full_ex := Ttiepie_hw_device_get_battery_time_to_full_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_device_get_battery_time_to_full_ex' ) );
      tiepie_hw_device_is_battery_charger_connected := Ttiepie_hw_device_is_battery_charger_connected( GetProcAddress( hLibTiePie , 'tiepie_hw_device_is_battery_charger_connected' ) );
      tiepie_hw_device_is_battery_charger_connected_ex := Ttiepie_hw_device_is_battery_charger_connected_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_device_is_battery_charger_connected_ex' ) );
      tiepie_hw_device_is_battery_charging := Ttiepie_hw_device_is_battery_charging( GetProcAddress( hLibTiePie , 'tiepie_hw_device_is_battery_charging' ) );
      tiepie_hw_device_is_battery_charging_ex := Ttiepie_hw_device_is_battery_charging_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_device_is_battery_charging_ex' ) );
      tiepie_hw_device_is_battery_broken := Ttiepie_hw_device_is_battery_broken( GetProcAddress( hLibTiePie , 'tiepie_hw_device_is_battery_broken' ) );
      tiepie_hw_device_is_battery_broken_ex := Ttiepie_hw_device_is_battery_broken_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_device_is_battery_broken_ex' ) );
      tiepie_hw_device_trigger_get_input_count := Ttiepie_hw_device_trigger_get_input_count( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_get_input_count' ) );
      tiepie_hw_device_trigger_get_input_index_by_id := Ttiepie_hw_device_trigger_get_input_index_by_id( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_get_input_index_by_id' ) );
      tiepie_hw_oscilloscope_trigger_input_is_triggered := Ttiepie_hw_oscilloscope_trigger_input_is_triggered( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_input_is_triggered' ) );
      tiepie_hw_device_trigger_input_get_enabled := Ttiepie_hw_device_trigger_input_get_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_input_get_enabled' ) );
      tiepie_hw_device_trigger_input_set_enabled := Ttiepie_hw_device_trigger_input_set_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_input_set_enabled' ) );
      tiepie_hw_device_trigger_input_get_kinds := Ttiepie_hw_device_trigger_input_get_kinds( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_input_get_kinds' ) );
      tiepie_hw_oscilloscope_trigger_input_get_kinds_ex := Ttiepie_hw_oscilloscope_trigger_input_get_kinds_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_input_get_kinds_ex' ) );
      tiepie_hw_device_trigger_input_get_kind := Ttiepie_hw_device_trigger_input_get_kind( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_input_get_kind' ) );
      tiepie_hw_device_trigger_input_set_kind := Ttiepie_hw_device_trigger_input_set_kind( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_input_set_kind' ) );
      tiepie_hw_device_trigger_input_is_available := Ttiepie_hw_device_trigger_input_is_available( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_input_is_available' ) );
      tiepie_hw_oscilloscope_trigger_input_is_available_ex := Ttiepie_hw_oscilloscope_trigger_input_is_available_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_input_is_available_ex' ) );
      tiepie_hw_device_trigger_input_get_id := Ttiepie_hw_device_trigger_input_get_id( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_input_get_id' ) );
      tiepie_hw_device_trigger_input_get_name := Ttiepie_hw_device_trigger_input_get_name( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_input_get_name' ) );
      tiepie_hw_device_trigger_get_output_count := Ttiepie_hw_device_trigger_get_output_count( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_get_output_count' ) );
      tiepie_hw_device_trigger_get_output_index_by_id := Ttiepie_hw_device_trigger_get_output_index_by_id( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_get_output_index_by_id' ) );
      tiepie_hw_device_trigger_output_get_enabled := Ttiepie_hw_device_trigger_output_get_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_output_get_enabled' ) );
      tiepie_hw_device_trigger_output_set_enabled := Ttiepie_hw_device_trigger_output_set_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_output_set_enabled' ) );
      tiepie_hw_device_trigger_output_get_events := Ttiepie_hw_device_trigger_output_get_events( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_output_get_events' ) );
      tiepie_hw_device_trigger_output_get_event := Ttiepie_hw_device_trigger_output_get_event( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_output_get_event' ) );
      tiepie_hw_device_trigger_output_set_event := Ttiepie_hw_device_trigger_output_set_event( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_output_set_event' ) );
      tiepie_hw_device_trigger_output_get_id := Ttiepie_hw_device_trigger_output_get_id( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_output_get_id' ) );
      tiepie_hw_device_trigger_output_get_name := Ttiepie_hw_device_trigger_output_get_name( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_output_get_name' ) );
      tiepie_hw_device_trigger_output_trigger := Ttiepie_hw_device_trigger_output_trigger( GetProcAddress( hLibTiePie , 'tiepie_hw_device_trigger_output_trigger' ) );
      tiepie_hw_oscilloscope_get_channel_count := Ttiepie_hw_oscilloscope_get_channel_count( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_channel_count' ) );
      tiepie_hw_oscilloscope_channel_is_available := Ttiepie_hw_oscilloscope_channel_is_available( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_is_available' ) );
      tiepie_hw_oscilloscope_channel_is_available_ex := Ttiepie_hw_oscilloscope_channel_is_available_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_is_available_ex' ) );
      tiepie_hw_oscilloscope_channel_get_connector_type := Ttiepie_hw_oscilloscope_channel_get_connector_type( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_connector_type' ) );
      tiepie_hw_oscilloscope_channel_is_differential := Ttiepie_hw_oscilloscope_channel_is_differential( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_is_differential' ) );
      tiepie_hw_oscilloscope_channel_is_isolated := Ttiepie_hw_oscilloscope_channel_is_isolated( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_is_isolated' ) );
      tiepie_hw_oscilloscope_channel_get_impedance := Ttiepie_hw_oscilloscope_channel_get_impedance( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_impedance' ) );
      tiepie_hw_oscilloscope_channel_get_bandwidths := Ttiepie_hw_oscilloscope_channel_get_bandwidths( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_bandwidths' ) );
      tiepie_hw_oscilloscope_channel_get_bandwidth := Ttiepie_hw_oscilloscope_channel_get_bandwidth( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_bandwidth' ) );
      tiepie_hw_oscilloscope_channel_set_bandwidth := Ttiepie_hw_oscilloscope_channel_set_bandwidth( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_set_bandwidth' ) );
      tiepie_hw_oscilloscope_channel_get_couplings := Ttiepie_hw_oscilloscope_channel_get_couplings( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_couplings' ) );
      tiepie_hw_oscilloscope_channel_get_coupling := Ttiepie_hw_oscilloscope_channel_get_coupling( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_coupling' ) );
      tiepie_hw_oscilloscope_channel_set_coupling := Ttiepie_hw_oscilloscope_channel_set_coupling( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_set_coupling' ) );
      tiepie_hw_oscilloscope_channel_get_enabled := Ttiepie_hw_oscilloscope_channel_get_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_enabled' ) );
      tiepie_hw_oscilloscope_channel_set_enabled := Ttiepie_hw_oscilloscope_channel_set_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_set_enabled' ) );
      tiepie_hw_oscilloscope_channel_get_auto_ranging := Ttiepie_hw_oscilloscope_channel_get_auto_ranging( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_auto_ranging' ) );
      tiepie_hw_oscilloscope_channel_set_auto_ranging := Ttiepie_hw_oscilloscope_channel_set_auto_ranging( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_set_auto_ranging' ) );
      tiepie_hw_oscilloscope_channel_get_ranges := Ttiepie_hw_oscilloscope_channel_get_ranges( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_ranges' ) );
      tiepie_hw_oscilloscope_channel_get_ranges_ex := Ttiepie_hw_oscilloscope_channel_get_ranges_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_ranges_ex' ) );
      tiepie_hw_oscilloscope_channel_get_range := Ttiepie_hw_oscilloscope_channel_get_range( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_range' ) );
      tiepie_hw_oscilloscope_channel_set_range := Ttiepie_hw_oscilloscope_channel_set_range( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_set_range' ) );
      tiepie_hw_oscilloscope_channel_has_safeground := Ttiepie_hw_oscilloscope_channel_has_safeground( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_has_safeground' ) );
      tiepie_hw_oscilloscope_channel_get_safeground_enabled := Ttiepie_hw_oscilloscope_channel_get_safeground_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_safeground_enabled' ) );
      tiepie_hw_oscilloscope_channel_set_safeground_enabled := Ttiepie_hw_oscilloscope_channel_set_safeground_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_set_safeground_enabled' ) );
      tiepie_hw_oscilloscope_channel_get_safeground_threshold_min := Ttiepie_hw_oscilloscope_channel_get_safeground_threshold_min( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_safeground_threshold_min' ) );
      tiepie_hw_oscilloscope_channel_get_safeground_threshold_max := Ttiepie_hw_oscilloscope_channel_get_safeground_threshold_max( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_safeground_threshold_max' ) );
      tiepie_hw_oscilloscope_channel_get_safeground_threshold := Ttiepie_hw_oscilloscope_channel_get_safeground_threshold( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_safeground_threshold' ) );
      tiepie_hw_oscilloscope_channel_set_safeground_threshold := Ttiepie_hw_oscilloscope_channel_set_safeground_threshold( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_set_safeground_threshold' ) );
      tiepie_hw_oscilloscope_channel_verify_safeground_threshold := Ttiepie_hw_oscilloscope_channel_verify_safeground_threshold( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_verify_safeground_threshold' ) );
      tiepie_hw_oscilloscope_channel_has_trigger := Ttiepie_hw_oscilloscope_channel_has_trigger( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_has_trigger' ) );
      tiepie_hw_oscilloscope_channel_has_trigger_ex := Ttiepie_hw_oscilloscope_channel_has_trigger_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_has_trigger_ex' ) );
      tiepie_hw_oscilloscope_channel_trigger_is_available := Ttiepie_hw_oscilloscope_channel_trigger_is_available( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_is_available' ) );
      tiepie_hw_oscilloscope_channel_trigger_is_available_ex := Ttiepie_hw_oscilloscope_channel_trigger_is_available_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_is_available_ex' ) );
      tiepie_hw_oscilloscope_channel_trigger_is_triggered := Ttiepie_hw_oscilloscope_channel_trigger_is_triggered( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_is_triggered' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_enabled := Ttiepie_hw_oscilloscope_channel_trigger_get_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_enabled' ) );
      tiepie_hw_oscilloscope_channel_trigger_set_enabled := Ttiepie_hw_oscilloscope_channel_trigger_set_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_set_enabled' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_kinds := Ttiepie_hw_oscilloscope_channel_trigger_get_kinds( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_kinds' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_kinds_ex := Ttiepie_hw_oscilloscope_channel_trigger_get_kinds_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_kinds_ex' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_kind := Ttiepie_hw_oscilloscope_channel_trigger_get_kind( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_kind' ) );
      tiepie_hw_oscilloscope_channel_trigger_set_kind := Ttiepie_hw_oscilloscope_channel_trigger_set_kind( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_set_kind' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_level_modes := Ttiepie_hw_oscilloscope_channel_trigger_get_level_modes( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_level_modes' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_level_mode := Ttiepie_hw_oscilloscope_channel_trigger_get_level_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_level_mode' ) );
      tiepie_hw_oscilloscope_channel_trigger_set_level_mode := Ttiepie_hw_oscilloscope_channel_trigger_set_level_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_set_level_mode' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_level_count := Ttiepie_hw_oscilloscope_channel_trigger_get_level_count( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_level_count' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_level := Ttiepie_hw_oscilloscope_channel_trigger_get_level( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_level' ) );
      tiepie_hw_oscilloscope_channel_trigger_set_level := Ttiepie_hw_oscilloscope_channel_trigger_set_level( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_set_level' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count := Ttiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_hysteresis := Ttiepie_hw_oscilloscope_channel_trigger_get_hysteresis( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_hysteresis' ) );
      tiepie_hw_oscilloscope_channel_trigger_set_hysteresis := Ttiepie_hw_oscilloscope_channel_trigger_set_hysteresis( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_set_hysteresis' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_conditions := Ttiepie_hw_oscilloscope_channel_trigger_get_conditions( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_conditions' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_conditions_ex := Ttiepie_hw_oscilloscope_channel_trigger_get_conditions_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_conditions_ex' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_condition := Ttiepie_hw_oscilloscope_channel_trigger_get_condition( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_condition' ) );
      tiepie_hw_oscilloscope_channel_trigger_set_condition := Ttiepie_hw_oscilloscope_channel_trigger_set_condition( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_set_condition' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_time_count := Ttiepie_hw_oscilloscope_channel_trigger_get_time_count( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_time_count' ) );
      tiepie_hw_oscilloscope_channel_trigger_get_time := Ttiepie_hw_oscilloscope_channel_trigger_get_time( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_get_time' ) );
      tiepie_hw_oscilloscope_channel_trigger_set_time := Ttiepie_hw_oscilloscope_channel_trigger_set_time( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_set_time' ) );
      tiepie_hw_oscilloscope_channel_trigger_verify_time := Ttiepie_hw_oscilloscope_channel_trigger_verify_time( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_verify_time' ) );
      tiepie_hw_oscilloscope_channel_trigger_verify_time_ex := Ttiepie_hw_oscilloscope_channel_trigger_verify_time_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_trigger_verify_time_ex' ) );
      tiepie_hw_oscilloscope_is_demo := Ttiepie_hw_oscilloscope_is_demo( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_demo' ) );
      tiepie_hw_oscilloscope_channel_demo_get_signal := Ttiepie_hw_oscilloscope_channel_demo_get_signal( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_get_signal' ) );
      tiepie_hw_oscilloscope_channel_demo_set_signal := Ttiepie_hw_oscilloscope_channel_demo_set_signal( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_set_signal' ) );
      tiepie_hw_oscilloscope_channel_demo_has_amplitude := Ttiepie_hw_oscilloscope_channel_demo_has_amplitude( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_has_amplitude' ) );
      tiepie_hw_oscilloscope_channel_demo_get_amplitude := Ttiepie_hw_oscilloscope_channel_demo_get_amplitude( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_get_amplitude' ) );
      tiepie_hw_oscilloscope_channel_demo_set_amplitude := Ttiepie_hw_oscilloscope_channel_demo_set_amplitude( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_set_amplitude' ) );
      tiepie_hw_oscilloscope_channel_demo_has_frequency := Ttiepie_hw_oscilloscope_channel_demo_has_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_has_frequency' ) );
      tiepie_hw_oscilloscope_channel_demo_get_frequency := Ttiepie_hw_oscilloscope_channel_demo_get_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_get_frequency' ) );
      tiepie_hw_oscilloscope_channel_demo_set_frequency := Ttiepie_hw_oscilloscope_channel_demo_set_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_set_frequency' ) );
      tiepie_hw_oscilloscope_channel_demo_has_offset := Ttiepie_hw_oscilloscope_channel_demo_has_offset( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_has_offset' ) );
      tiepie_hw_oscilloscope_channel_demo_get_offset := Ttiepie_hw_oscilloscope_channel_demo_get_offset( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_get_offset' ) );
      tiepie_hw_oscilloscope_channel_demo_set_offset := Ttiepie_hw_oscilloscope_channel_demo_set_offset( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_set_offset' ) );
      tiepie_hw_oscilloscope_channel_demo_has_symmetry := Ttiepie_hw_oscilloscope_channel_demo_has_symmetry( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_has_symmetry' ) );
      tiepie_hw_oscilloscope_channel_demo_get_symmetry := Ttiepie_hw_oscilloscope_channel_demo_get_symmetry( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_get_symmetry' ) );
      tiepie_hw_oscilloscope_channel_demo_set_symmetry := Ttiepie_hw_oscilloscope_channel_demo_set_symmetry( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_set_symmetry' ) );
      tiepie_hw_oscilloscope_channel_demo_has_phase := Ttiepie_hw_oscilloscope_channel_demo_has_phase( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_has_phase' ) );
      tiepie_hw_oscilloscope_channel_demo_get_phase := Ttiepie_hw_oscilloscope_channel_demo_get_phase( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_get_phase' ) );
      tiepie_hw_oscilloscope_channel_demo_set_phase := Ttiepie_hw_oscilloscope_channel_demo_set_phase( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_set_phase' ) );
      tiepie_hw_oscilloscope_channel_demo_has_noise_enabled := Ttiepie_hw_oscilloscope_channel_demo_has_noise_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_has_noise_enabled' ) );
      tiepie_hw_oscilloscope_channel_demo_get_noise_enabled := Ttiepie_hw_oscilloscope_channel_demo_get_noise_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_get_noise_enabled' ) );
      tiepie_hw_oscilloscope_channel_demo_set_noise_enabled := Ttiepie_hw_oscilloscope_channel_demo_set_noise_enabled( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_demo_set_noise_enabled' ) );
      tiepie_hw_oscilloscope_get_data := Ttiepie_hw_oscilloscope_get_data( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data' ) );
      tiepie_hw_oscilloscope_get_data_1ch := Ttiepie_hw_oscilloscope_get_data_1ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_1ch' ) );
      tiepie_hw_oscilloscope_get_data_2ch := Ttiepie_hw_oscilloscope_get_data_2ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_2ch' ) );
      tiepie_hw_oscilloscope_get_data_3ch := Ttiepie_hw_oscilloscope_get_data_3ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_3ch' ) );
      tiepie_hw_oscilloscope_get_data_4ch := Ttiepie_hw_oscilloscope_get_data_4ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_4ch' ) );
      tiepie_hw_oscilloscope_get_data_5ch := Ttiepie_hw_oscilloscope_get_data_5ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_5ch' ) );
      tiepie_hw_oscilloscope_get_data_6ch := Ttiepie_hw_oscilloscope_get_data_6ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_6ch' ) );
      tiepie_hw_oscilloscope_get_data_7ch := Ttiepie_hw_oscilloscope_get_data_7ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_7ch' ) );
      tiepie_hw_oscilloscope_get_data_8ch := Ttiepie_hw_oscilloscope_get_data_8ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_8ch' ) );
      tiepie_hw_oscilloscope_get_valid_pre_sample_count := Ttiepie_hw_oscilloscope_get_valid_pre_sample_count( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_valid_pre_sample_count' ) );
      tiepie_hw_oscilloscope_channel_get_data_value_range := Ttiepie_hw_oscilloscope_channel_get_data_value_range( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_data_value_range' ) );
      tiepie_hw_oscilloscope_channel_get_data_value_min := Ttiepie_hw_oscilloscope_channel_get_data_value_min( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_data_value_min' ) );
      tiepie_hw_oscilloscope_channel_get_data_value_max := Ttiepie_hw_oscilloscope_channel_get_data_value_max( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_data_value_max' ) );
      tiepie_hw_oscilloscope_get_data_raw := Ttiepie_hw_oscilloscope_get_data_raw( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw' ) );
      tiepie_hw_oscilloscope_get_data_raw_1ch := Ttiepie_hw_oscilloscope_get_data_raw_1ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw_1ch' ) );
      tiepie_hw_oscilloscope_get_data_raw_2ch := Ttiepie_hw_oscilloscope_get_data_raw_2ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw_2ch' ) );
      tiepie_hw_oscilloscope_get_data_raw_3ch := Ttiepie_hw_oscilloscope_get_data_raw_3ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw_3ch' ) );
      tiepie_hw_oscilloscope_get_data_raw_4ch := Ttiepie_hw_oscilloscope_get_data_raw_4ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw_4ch' ) );
      tiepie_hw_oscilloscope_get_data_raw_5ch := Ttiepie_hw_oscilloscope_get_data_raw_5ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw_5ch' ) );
      tiepie_hw_oscilloscope_get_data_raw_6ch := Ttiepie_hw_oscilloscope_get_data_raw_6ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw_6ch' ) );
      tiepie_hw_oscilloscope_get_data_raw_7ch := Ttiepie_hw_oscilloscope_get_data_raw_7ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw_7ch' ) );
      tiepie_hw_oscilloscope_get_data_raw_8ch := Ttiepie_hw_oscilloscope_get_data_raw_8ch( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_data_raw_8ch' ) );
      tiepie_hw_oscilloscope_channel_get_data_raw_type := Ttiepie_hw_oscilloscope_channel_get_data_raw_type( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_data_raw_type' ) );
      tiepie_hw_oscilloscope_channel_get_data_raw_value_range := Ttiepie_hw_oscilloscope_channel_get_data_raw_value_range( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_data_raw_value_range' ) );
      tiepie_hw_oscilloscope_channel_get_data_raw_value_min := Ttiepie_hw_oscilloscope_channel_get_data_raw_value_min( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_data_raw_value_min' ) );
      tiepie_hw_oscilloscope_channel_get_data_raw_value_zero := Ttiepie_hw_oscilloscope_channel_get_data_raw_value_zero( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_data_raw_value_zero' ) );
      tiepie_hw_oscilloscope_channel_get_data_raw_value_max := Ttiepie_hw_oscilloscope_channel_get_data_raw_value_max( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_get_data_raw_value_max' ) );
      tiepie_hw_oscilloscope_channel_is_range_max_reachable := Ttiepie_hw_oscilloscope_channel_is_range_max_reachable( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_is_range_max_reachable' ) );
      tiepie_hw_oscilloscope_start := Ttiepie_hw_oscilloscope_start( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_start' ) );
      tiepie_hw_oscilloscope_stop := Ttiepie_hw_oscilloscope_stop( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_stop' ) );
      tiepie_hw_oscilloscope_force_trigger := Ttiepie_hw_oscilloscope_force_trigger( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_force_trigger' ) );
      tiepie_hw_oscilloscope_get_measure_modes := Ttiepie_hw_oscilloscope_get_measure_modes( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_measure_modes' ) );
      tiepie_hw_oscilloscope_get_measure_mode := Ttiepie_hw_oscilloscope_get_measure_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_measure_mode' ) );
      tiepie_hw_oscilloscope_set_measure_mode := Ttiepie_hw_oscilloscope_set_measure_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_measure_mode' ) );
      tiepie_hw_oscilloscope_is_running := Ttiepie_hw_oscilloscope_is_running( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_running' ) );
      tiepie_hw_oscilloscope_is_triggered := Ttiepie_hw_oscilloscope_is_triggered( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_triggered' ) );
      tiepie_hw_oscilloscope_is_timeout_triggered := Ttiepie_hw_oscilloscope_is_timeout_triggered( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_timeout_triggered' ) );
      tiepie_hw_oscilloscope_is_force_triggered := Ttiepie_hw_oscilloscope_is_force_triggered( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_force_triggered' ) );
      tiepie_hw_oscilloscope_is_data_ready := Ttiepie_hw_oscilloscope_is_data_ready( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_data_ready' ) );
      tiepie_hw_oscilloscope_is_data_overflow := Ttiepie_hw_oscilloscope_is_data_overflow( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_data_overflow' ) );
      tiepie_hw_oscilloscope_get_auto_resolution_modes := Ttiepie_hw_oscilloscope_get_auto_resolution_modes( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_auto_resolution_modes' ) );
      tiepie_hw_oscilloscope_get_auto_resolution_mode := Ttiepie_hw_oscilloscope_get_auto_resolution_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_auto_resolution_mode' ) );
      tiepie_hw_oscilloscope_set_auto_resolution_mode := Ttiepie_hw_oscilloscope_set_auto_resolution_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_auto_resolution_mode' ) );
      tiepie_hw_oscilloscope_get_resolutions := Ttiepie_hw_oscilloscope_get_resolutions( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_resolutions' ) );
      tiepie_hw_oscilloscope_get_resolution := Ttiepie_hw_oscilloscope_get_resolution( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_resolution' ) );
      tiepie_hw_oscilloscope_set_resolution := Ttiepie_hw_oscilloscope_set_resolution( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_resolution' ) );
      tiepie_hw_oscilloscope_is_resolution_enhanced := Ttiepie_hw_oscilloscope_is_resolution_enhanced( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_resolution_enhanced' ) );
      tiepie_hw_oscilloscope_is_resolution_enhanced_ex := Ttiepie_hw_oscilloscope_is_resolution_enhanced_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_resolution_enhanced_ex' ) );
      tiepie_hw_oscilloscope_get_clock_sources := Ttiepie_hw_oscilloscope_get_clock_sources( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_sources' ) );
      tiepie_hw_oscilloscope_get_clock_source := Ttiepie_hw_oscilloscope_get_clock_source( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_source' ) );
      tiepie_hw_oscilloscope_set_clock_source := Ttiepie_hw_oscilloscope_set_clock_source( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_clock_source' ) );
      tiepie_hw_oscilloscope_get_clock_source_frequencies := Ttiepie_hw_oscilloscope_get_clock_source_frequencies( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_source_frequencies' ) );
      tiepie_hw_oscilloscope_get_clock_source_frequencies_ex := Ttiepie_hw_oscilloscope_get_clock_source_frequencies_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_source_frequencies_ex' ) );
      tiepie_hw_oscilloscope_get_clock_source_frequency := Ttiepie_hw_oscilloscope_get_clock_source_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_source_frequency' ) );
      tiepie_hw_oscilloscope_set_clock_source_frequency := Ttiepie_hw_oscilloscope_set_clock_source_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_clock_source_frequency' ) );
      tiepie_hw_oscilloscope_get_clock_outputs := Ttiepie_hw_oscilloscope_get_clock_outputs( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_outputs' ) );
      tiepie_hw_oscilloscope_get_clock_output := Ttiepie_hw_oscilloscope_get_clock_output( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_output' ) );
      tiepie_hw_oscilloscope_set_clock_output := Ttiepie_hw_oscilloscope_set_clock_output( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_clock_output' ) );
      tiepie_hw_oscilloscope_get_clock_output_frequencies := Ttiepie_hw_oscilloscope_get_clock_output_frequencies( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_output_frequencies' ) );
      tiepie_hw_oscilloscope_get_clock_output_frequencies_ex := Ttiepie_hw_oscilloscope_get_clock_output_frequencies_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_output_frequencies_ex' ) );
      tiepie_hw_oscilloscope_get_clock_output_frequency := Ttiepie_hw_oscilloscope_get_clock_output_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_clock_output_frequency' ) );
      tiepie_hw_oscilloscope_set_clock_output_frequency := Ttiepie_hw_oscilloscope_set_clock_output_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_clock_output_frequency' ) );
      tiepie_hw_oscilloscope_get_sample_rate_max := Ttiepie_hw_oscilloscope_get_sample_rate_max( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_sample_rate_max' ) );
      tiepie_hw_oscilloscope_get_sample_rate := Ttiepie_hw_oscilloscope_get_sample_rate( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_sample_rate' ) );
      tiepie_hw_oscilloscope_set_sample_rate := Ttiepie_hw_oscilloscope_set_sample_rate( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_sample_rate' ) );
      tiepie_hw_oscilloscope_verify_sample_rate := Ttiepie_hw_oscilloscope_verify_sample_rate( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_verify_sample_rate' ) );
      tiepie_hw_oscilloscope_verify_sample_rate_ex := Ttiepie_hw_oscilloscope_verify_sample_rate_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_verify_sample_rate_ex' ) );
      tiepie_hw_oscilloscope_verify_sample_rates_ex := Ttiepie_hw_oscilloscope_verify_sample_rates_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_verify_sample_rates_ex' ) );
      tiepie_hw_oscilloscope_get_record_length_max := Ttiepie_hw_oscilloscope_get_record_length_max( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_record_length_max' ) );
      tiepie_hw_oscilloscope_get_record_length_max_ex := Ttiepie_hw_oscilloscope_get_record_length_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_record_length_max_ex' ) );
      tiepie_hw_oscilloscope_get_record_length := Ttiepie_hw_oscilloscope_get_record_length( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_record_length' ) );
      tiepie_hw_oscilloscope_set_record_length := Ttiepie_hw_oscilloscope_set_record_length( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_record_length' ) );
      tiepie_hw_oscilloscope_verify_record_length := Ttiepie_hw_oscilloscope_verify_record_length( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_verify_record_length' ) );
      tiepie_hw_oscilloscope_verify_record_length_ex := Ttiepie_hw_oscilloscope_verify_record_length_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_verify_record_length_ex' ) );
      tiepie_hw_oscilloscope_get_pre_sample_ratio := Ttiepie_hw_oscilloscope_get_pre_sample_ratio( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_pre_sample_ratio' ) );
      tiepie_hw_oscilloscope_set_pre_sample_ratio := Ttiepie_hw_oscilloscope_set_pre_sample_ratio( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_pre_sample_ratio' ) );
      tiepie_hw_oscilloscope_get_segment_count_max := Ttiepie_hw_oscilloscope_get_segment_count_max( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_segment_count_max' ) );
      tiepie_hw_oscilloscope_get_segment_count_max_ex := Ttiepie_hw_oscilloscope_get_segment_count_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_segment_count_max_ex' ) );
      tiepie_hw_oscilloscope_get_segment_count := Ttiepie_hw_oscilloscope_get_segment_count( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_segment_count' ) );
      tiepie_hw_oscilloscope_set_segment_count := Ttiepie_hw_oscilloscope_set_segment_count( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_segment_count' ) );
      tiepie_hw_oscilloscope_verify_segment_count := Ttiepie_hw_oscilloscope_verify_segment_count( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_verify_segment_count' ) );
      tiepie_hw_oscilloscope_verify_segment_count_ex := Ttiepie_hw_oscilloscope_verify_segment_count_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_verify_segment_count_ex' ) );
      tiepie_hw_oscilloscope_has_trigger := Ttiepie_hw_oscilloscope_has_trigger( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_has_trigger' ) );
      tiepie_hw_oscilloscope_has_trigger_ex := Ttiepie_hw_oscilloscope_has_trigger_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_has_trigger_ex' ) );
      tiepie_hw_oscilloscope_trigger_get_timeout := Ttiepie_hw_oscilloscope_trigger_get_timeout( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_get_timeout' ) );
      tiepie_hw_oscilloscope_trigger_set_timeout := Ttiepie_hw_oscilloscope_trigger_set_timeout( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_set_timeout' ) );
      tiepie_hw_oscilloscope_trigger_verify_timeout := Ttiepie_hw_oscilloscope_trigger_verify_timeout( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_verify_timeout' ) );
      tiepie_hw_oscilloscope_trigger_verify_timeout_ex := Ttiepie_hw_oscilloscope_trigger_verify_timeout_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_verify_timeout_ex' ) );
      tiepie_hw_oscilloscope_trigger_has_delay := Ttiepie_hw_oscilloscope_trigger_has_delay( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_has_delay' ) );
      tiepie_hw_oscilloscope_trigger_has_delay_ex := Ttiepie_hw_oscilloscope_trigger_has_delay_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_has_delay_ex' ) );
      tiepie_hw_oscilloscope_trigger_get_delay_max := Ttiepie_hw_oscilloscope_trigger_get_delay_max( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_get_delay_max' ) );
      tiepie_hw_oscilloscope_trigger_get_delay_max_ex := Ttiepie_hw_oscilloscope_trigger_get_delay_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_get_delay_max_ex' ) );
      tiepie_hw_oscilloscope_trigger_get_delay := Ttiepie_hw_oscilloscope_trigger_get_delay( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_get_delay' ) );
      tiepie_hw_oscilloscope_trigger_set_delay := Ttiepie_hw_oscilloscope_trigger_set_delay( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_set_delay' ) );
      tiepie_hw_oscilloscope_trigger_verify_delay := Ttiepie_hw_oscilloscope_trigger_verify_delay( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_verify_delay' ) );
      tiepie_hw_oscilloscope_trigger_verify_delay_ex := Ttiepie_hw_oscilloscope_trigger_verify_delay_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_trigger_verify_delay_ex' ) );
      tiepie_hw_oscilloscope_has_presamples_valid := Ttiepie_hw_oscilloscope_has_presamples_valid( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_has_presamples_valid' ) );
      tiepie_hw_oscilloscope_has_presamples_valid_ex := Ttiepie_hw_oscilloscope_has_presamples_valid_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_has_presamples_valid_ex' ) );
      tiepie_hw_oscilloscope_get_presamples_valid := Ttiepie_hw_oscilloscope_get_presamples_valid( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_presamples_valid' ) );
      tiepie_hw_oscilloscope_set_presamples_valid := Ttiepie_hw_oscilloscope_set_presamples_valid( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_set_presamples_valid' ) );
      tiepie_hw_oscilloscope_has_sureconnect := Ttiepie_hw_oscilloscope_has_sureconnect( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_has_sureconnect' ) );
      tiepie_hw_oscilloscope_channel_has_sureconnect := Ttiepie_hw_oscilloscope_channel_has_sureconnect( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_channel_has_sureconnect' ) );
      tiepie_hw_oscilloscope_start_sureconnect := Ttiepie_hw_oscilloscope_start_sureconnect( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_start_sureconnect' ) );
      tiepie_hw_oscilloscope_start_sureconnect_ex := Ttiepie_hw_oscilloscope_start_sureconnect_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_start_sureconnect_ex' ) );
      tiepie_hw_oscilloscope_is_sureconnect_completed := Ttiepie_hw_oscilloscope_is_sureconnect_completed( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_is_sureconnect_completed' ) );
      tiepie_hw_oscilloscope_get_sureconnect_data := Ttiepie_hw_oscilloscope_get_sureconnect_data( GetProcAddress( hLibTiePie , 'tiepie_hw_oscilloscope_get_sureconnect_data' ) );
      tiepie_hw_generator_get_connector_type := Ttiepie_hw_generator_get_connector_type( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_connector_type' ) );
      tiepie_hw_generator_is_differential := Ttiepie_hw_generator_is_differential( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_is_differential' ) );
      tiepie_hw_generator_get_impedance := Ttiepie_hw_generator_get_impedance( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_impedance' ) );
      tiepie_hw_generator_get_resolution := Ttiepie_hw_generator_get_resolution( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_resolution' ) );
      tiepie_hw_generator_get_output_value_min := Ttiepie_hw_generator_get_output_value_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_output_value_min' ) );
      tiepie_hw_generator_get_output_value_max := Ttiepie_hw_generator_get_output_value_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_output_value_max' ) );
      tiepie_hw_generator_get_output_value_min_max := Ttiepie_hw_generator_get_output_value_min_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_output_value_min_max' ) );
      tiepie_hw_generator_is_controllable := Ttiepie_hw_generator_is_controllable( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_is_controllable' ) );
      tiepie_hw_generator_is_running := Ttiepie_hw_generator_is_running( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_is_running' ) );
      tiepie_hw_generator_get_status := Ttiepie_hw_generator_get_status( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_status' ) );
      tiepie_hw_generator_get_output_enable := Ttiepie_hw_generator_get_output_enable( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_output_enable' ) );
      tiepie_hw_generator_set_output_enable := Ttiepie_hw_generator_set_output_enable( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_output_enable' ) );
      tiepie_hw_generator_has_output_invert := Ttiepie_hw_generator_has_output_invert( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_output_invert' ) );
      tiepie_hw_generator_get_output_invert := Ttiepie_hw_generator_get_output_invert( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_output_invert' ) );
      tiepie_hw_generator_set_output_invert := Ttiepie_hw_generator_set_output_invert( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_output_invert' ) );
      tiepie_hw_generator_start := Ttiepie_hw_generator_start( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_start' ) );
      tiepie_hw_generator_stop := Ttiepie_hw_generator_stop( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_stop' ) );
      tiepie_hw_generator_get_signal_types := Ttiepie_hw_generator_get_signal_types( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_signal_types' ) );
      tiepie_hw_generator_get_signal_type := Ttiepie_hw_generator_get_signal_type( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_signal_type' ) );
      tiepie_hw_generator_set_signal_type := Ttiepie_hw_generator_set_signal_type( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_signal_type' ) );
      tiepie_hw_generator_has_amplitude := Ttiepie_hw_generator_has_amplitude( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_amplitude' ) );
      tiepie_hw_generator_has_amplitude_ex := Ttiepie_hw_generator_has_amplitude_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_amplitude_ex' ) );
      tiepie_hw_generator_get_amplitude_min := Ttiepie_hw_generator_get_amplitude_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_amplitude_min' ) );
      tiepie_hw_generator_get_amplitude_max := Ttiepie_hw_generator_get_amplitude_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_amplitude_max' ) );
      tiepie_hw_generator_get_amplitude_min_max_ex := Ttiepie_hw_generator_get_amplitude_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_amplitude_min_max_ex' ) );
      tiepie_hw_generator_get_amplitude := Ttiepie_hw_generator_get_amplitude( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_amplitude' ) );
      tiepie_hw_generator_set_amplitude := Ttiepie_hw_generator_set_amplitude( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_amplitude' ) );
      tiepie_hw_generator_verify_amplitude := Ttiepie_hw_generator_verify_amplitude( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_amplitude' ) );
      tiepie_hw_generator_verify_amplitude_ex := Ttiepie_hw_generator_verify_amplitude_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_amplitude_ex' ) );
      tiepie_hw_generator_get_amplitude_ranges := Ttiepie_hw_generator_get_amplitude_ranges( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_amplitude_ranges' ) );
      tiepie_hw_generator_get_amplitude_range := Ttiepie_hw_generator_get_amplitude_range( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_amplitude_range' ) );
      tiepie_hw_generator_set_amplitude_range := Ttiepie_hw_generator_set_amplitude_range( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_amplitude_range' ) );
      tiepie_hw_generator_get_amplitude_auto_ranging := Ttiepie_hw_generator_get_amplitude_auto_ranging( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_amplitude_auto_ranging' ) );
      tiepie_hw_generator_set_amplitude_auto_ranging := Ttiepie_hw_generator_set_amplitude_auto_ranging( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_amplitude_auto_ranging' ) );
      tiepie_hw_generator_has_offset := Ttiepie_hw_generator_has_offset( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_offset' ) );
      tiepie_hw_generator_has_offset_ex := Ttiepie_hw_generator_has_offset_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_offset_ex' ) );
      tiepie_hw_generator_get_offset_min := Ttiepie_hw_generator_get_offset_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_offset_min' ) );
      tiepie_hw_generator_get_offset_max := Ttiepie_hw_generator_get_offset_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_offset_max' ) );
      tiepie_hw_generator_get_offset_min_max_ex := Ttiepie_hw_generator_get_offset_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_offset_min_max_ex' ) );
      tiepie_hw_generator_get_offset := Ttiepie_hw_generator_get_offset( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_offset' ) );
      tiepie_hw_generator_set_offset := Ttiepie_hw_generator_set_offset( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_offset' ) );
      tiepie_hw_generator_verify_offset := Ttiepie_hw_generator_verify_offset( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_offset' ) );
      tiepie_hw_generator_verify_offset_ex := Ttiepie_hw_generator_verify_offset_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_offset_ex' ) );
      tiepie_hw_generator_get_frequency_modes := Ttiepie_hw_generator_get_frequency_modes( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_frequency_modes' ) );
      tiepie_hw_generator_get_frequency_modes_ex := Ttiepie_hw_generator_get_frequency_modes_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_frequency_modes_ex' ) );
      tiepie_hw_generator_get_frequency_mode := Ttiepie_hw_generator_get_frequency_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_frequency_mode' ) );
      tiepie_hw_generator_set_frequency_mode := Ttiepie_hw_generator_set_frequency_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_frequency_mode' ) );
      tiepie_hw_generator_has_frequency := Ttiepie_hw_generator_has_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_frequency' ) );
      tiepie_hw_generator_has_frequency_ex := Ttiepie_hw_generator_has_frequency_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_frequency_ex' ) );
      tiepie_hw_generator_get_frequency_min := Ttiepie_hw_generator_get_frequency_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_frequency_min' ) );
      tiepie_hw_generator_get_frequency_max := Ttiepie_hw_generator_get_frequency_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_frequency_max' ) );
      tiepie_hw_generator_get_frequency_min_max := Ttiepie_hw_generator_get_frequency_min_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_frequency_min_max' ) );
      tiepie_hw_generator_get_frequency_min_max_ex := Ttiepie_hw_generator_get_frequency_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_frequency_min_max_ex' ) );
      tiepie_hw_generator_get_frequency := Ttiepie_hw_generator_get_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_frequency' ) );
      tiepie_hw_generator_set_frequency := Ttiepie_hw_generator_set_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_frequency' ) );
      tiepie_hw_generator_verify_frequency := Ttiepie_hw_generator_verify_frequency( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_frequency' ) );
      tiepie_hw_generator_verify_frequency_ex := Ttiepie_hw_generator_verify_frequency_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_frequency_ex' ) );
      tiepie_hw_generator_has_phase := Ttiepie_hw_generator_has_phase( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_phase' ) );
      tiepie_hw_generator_has_phase_ex := Ttiepie_hw_generator_has_phase_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_phase_ex' ) );
      tiepie_hw_generator_get_phase_min := Ttiepie_hw_generator_get_phase_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_phase_min' ) );
      tiepie_hw_generator_get_phase_max := Ttiepie_hw_generator_get_phase_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_phase_max' ) );
      tiepie_hw_generator_get_phase_min_max_ex := Ttiepie_hw_generator_get_phase_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_phase_min_max_ex' ) );
      tiepie_hw_generator_get_phase := Ttiepie_hw_generator_get_phase( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_phase' ) );
      tiepie_hw_generator_set_phase := Ttiepie_hw_generator_set_phase( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_phase' ) );
      tiepie_hw_generator_verify_phase := Ttiepie_hw_generator_verify_phase( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_phase' ) );
      tiepie_hw_generator_verify_phase_ex := Ttiepie_hw_generator_verify_phase_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_phase_ex' ) );
      tiepie_hw_generator_has_symmetry := Ttiepie_hw_generator_has_symmetry( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_symmetry' ) );
      tiepie_hw_generator_has_symmetry_ex := Ttiepie_hw_generator_has_symmetry_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_symmetry_ex' ) );
      tiepie_hw_generator_get_symmetry_min := Ttiepie_hw_generator_get_symmetry_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_symmetry_min' ) );
      tiepie_hw_generator_get_symmetry_max := Ttiepie_hw_generator_get_symmetry_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_symmetry_max' ) );
      tiepie_hw_generator_get_symmetry_min_max_ex := Ttiepie_hw_generator_get_symmetry_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_symmetry_min_max_ex' ) );
      tiepie_hw_generator_get_symmetry := Ttiepie_hw_generator_get_symmetry( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_symmetry' ) );
      tiepie_hw_generator_set_symmetry := Ttiepie_hw_generator_set_symmetry( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_symmetry' ) );
      tiepie_hw_generator_verify_symmetry := Ttiepie_hw_generator_verify_symmetry( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_symmetry' ) );
      tiepie_hw_generator_verify_symmetry_ex := Ttiepie_hw_generator_verify_symmetry_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_symmetry_ex' ) );
      tiepie_hw_generator_has_width := Ttiepie_hw_generator_has_width( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_width' ) );
      tiepie_hw_generator_has_width_ex := Ttiepie_hw_generator_has_width_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_width_ex' ) );
      tiepie_hw_generator_get_width_min := Ttiepie_hw_generator_get_width_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_width_min' ) );
      tiepie_hw_generator_get_width_max := Ttiepie_hw_generator_get_width_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_width_max' ) );
      tiepie_hw_generator_get_width_min_max_ex := Ttiepie_hw_generator_get_width_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_width_min_max_ex' ) );
      tiepie_hw_generator_get_width := Ttiepie_hw_generator_get_width( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_width' ) );
      tiepie_hw_generator_set_width := Ttiepie_hw_generator_set_width( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_width' ) );
      tiepie_hw_generator_verify_width := Ttiepie_hw_generator_verify_width( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_width' ) );
      tiepie_hw_generator_verify_width_ex := Ttiepie_hw_generator_verify_width_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_width_ex' ) );
      tiepie_hw_generator_has_edge_time := Ttiepie_hw_generator_has_edge_time( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_edge_time' ) );
      tiepie_hw_generator_has_edge_time_ex := Ttiepie_hw_generator_has_edge_time_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_edge_time_ex' ) );
      tiepie_hw_generator_get_leading_edge_time_min := Ttiepie_hw_generator_get_leading_edge_time_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_leading_edge_time_min' ) );
      tiepie_hw_generator_get_leading_edge_time_max := Ttiepie_hw_generator_get_leading_edge_time_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_leading_edge_time_max' ) );
      tiepie_hw_generator_get_leading_edge_time_min_max_ex := Ttiepie_hw_generator_get_leading_edge_time_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_leading_edge_time_min_max_ex' ) );
      tiepie_hw_generator_get_leading_edge_time := Ttiepie_hw_generator_get_leading_edge_time( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_leading_edge_time' ) );
      tiepie_hw_generator_set_leading_edge_time := Ttiepie_hw_generator_set_leading_edge_time( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_leading_edge_time' ) );
      tiepie_hw_generator_verify_leading_edge_time := Ttiepie_hw_generator_verify_leading_edge_time( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_leading_edge_time' ) );
      tiepie_hw_generator_verify_leading_edge_time_ex := Ttiepie_hw_generator_verify_leading_edge_time_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_leading_edge_time_ex' ) );
      tiepie_hw_generator_get_trailing_edge_time_min := Ttiepie_hw_generator_get_trailing_edge_time_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_trailing_edge_time_min' ) );
      tiepie_hw_generator_get_trailing_edge_time_max := Ttiepie_hw_generator_get_trailing_edge_time_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_trailing_edge_time_max' ) );
      tiepie_hw_generator_get_trailing_edge_time_min_max_ex := Ttiepie_hw_generator_get_trailing_edge_time_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_trailing_edge_time_min_max_ex' ) );
      tiepie_hw_generator_get_trailing_edge_time := Ttiepie_hw_generator_get_trailing_edge_time( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_trailing_edge_time' ) );
      tiepie_hw_generator_set_trailing_edge_time := Ttiepie_hw_generator_set_trailing_edge_time( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_trailing_edge_time' ) );
      tiepie_hw_generator_verify_trailing_edge_time := Ttiepie_hw_generator_verify_trailing_edge_time( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_trailing_edge_time' ) );
      tiepie_hw_generator_verify_trailing_edge_time_ex := Ttiepie_hw_generator_verify_trailing_edge_time_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_trailing_edge_time_ex' ) );
      tiepie_hw_generator_has_data := Ttiepie_hw_generator_has_data( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_data' ) );
      tiepie_hw_generator_has_data_ex := Ttiepie_hw_generator_has_data_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_has_data_ex' ) );
      tiepie_hw_generator_get_data_length_min := Ttiepie_hw_generator_get_data_length_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_length_min' ) );
      tiepie_hw_generator_get_data_length_max := Ttiepie_hw_generator_get_data_length_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_length_max' ) );
      tiepie_hw_generator_get_data_length_min_max_ex := Ttiepie_hw_generator_get_data_length_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_length_min_max_ex' ) );
      tiepie_hw_generator_get_data_length := Ttiepie_hw_generator_get_data_length( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_length' ) );
      tiepie_hw_generator_verify_data_length := Ttiepie_hw_generator_verify_data_length( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_data_length' ) );
      tiepie_hw_generator_verify_data_length_ex := Ttiepie_hw_generator_verify_data_length_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_data_length_ex' ) );
      tiepie_hw_generator_set_data := Ttiepie_hw_generator_set_data( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_data' ) );
      tiepie_hw_generator_set_data_ex := Ttiepie_hw_generator_set_data_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_data_ex' ) );
      tiepie_hw_generator_get_data_raw_type := Ttiepie_hw_generator_get_data_raw_type( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_raw_type' ) );
      tiepie_hw_generator_get_data_raw_value_range := Ttiepie_hw_generator_get_data_raw_value_range( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_raw_value_range' ) );
      tiepie_hw_generator_get_data_raw_value_min := Ttiepie_hw_generator_get_data_raw_value_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_raw_value_min' ) );
      tiepie_hw_generator_get_data_raw_value_zero := Ttiepie_hw_generator_get_data_raw_value_zero( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_raw_value_zero' ) );
      tiepie_hw_generator_get_data_raw_value_max := Ttiepie_hw_generator_get_data_raw_value_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_data_raw_value_max' ) );
      tiepie_hw_generator_set_data_raw := Ttiepie_hw_generator_set_data_raw( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_data_raw' ) );
      tiepie_hw_generator_set_data_raw_ex := Ttiepie_hw_generator_set_data_raw_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_data_raw_ex' ) );
      tiepie_hw_generator_get_modes := Ttiepie_hw_generator_get_modes( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_modes' ) );
      tiepie_hw_generator_get_modes_ex := Ttiepie_hw_generator_get_modes_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_modes_ex' ) );
      tiepie_hw_generator_get_modes_native := Ttiepie_hw_generator_get_modes_native( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_modes_native' ) );
      tiepie_hw_generator_get_mode := Ttiepie_hw_generator_get_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_mode' ) );
      tiepie_hw_generator_set_mode := Ttiepie_hw_generator_set_mode( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_mode' ) );
      tiepie_hw_generator_is_burst_active := Ttiepie_hw_generator_is_burst_active( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_is_burst_active' ) );
      tiepie_hw_generator_get_burst_count_min := Ttiepie_hw_generator_get_burst_count_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_count_min' ) );
      tiepie_hw_generator_get_burst_count_max := Ttiepie_hw_generator_get_burst_count_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_count_max' ) );
      tiepie_hw_generator_get_burst_count_min_max_ex := Ttiepie_hw_generator_get_burst_count_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_count_min_max_ex' ) );
      tiepie_hw_generator_get_burst_count := Ttiepie_hw_generator_get_burst_count( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_count' ) );
      tiepie_hw_generator_set_burst_count := Ttiepie_hw_generator_set_burst_count( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_burst_count' ) );
      tiepie_hw_generator_get_burst_sample_count_min := Ttiepie_hw_generator_get_burst_sample_count_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_sample_count_min' ) );
      tiepie_hw_generator_get_burst_sample_count_max := Ttiepie_hw_generator_get_burst_sample_count_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_sample_count_max' ) );
      tiepie_hw_generator_get_burst_sample_count_min_max_ex := Ttiepie_hw_generator_get_burst_sample_count_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_sample_count_min_max_ex' ) );
      tiepie_hw_generator_get_burst_sample_count := Ttiepie_hw_generator_get_burst_sample_count( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_sample_count' ) );
      tiepie_hw_generator_set_burst_sample_count := Ttiepie_hw_generator_set_burst_sample_count( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_burst_sample_count' ) );
      tiepie_hw_generator_get_burst_segment_count_min := Ttiepie_hw_generator_get_burst_segment_count_min( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_segment_count_min' ) );
      tiepie_hw_generator_get_burst_segment_count_max := Ttiepie_hw_generator_get_burst_segment_count_max( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_segment_count_max' ) );
      tiepie_hw_generator_get_burst_segment_count_min_max_ex := Ttiepie_hw_generator_get_burst_segment_count_min_max_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_segment_count_min_max_ex' ) );
      tiepie_hw_generator_get_burst_segment_count := Ttiepie_hw_generator_get_burst_segment_count( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_get_burst_segment_count' ) );
      tiepie_hw_generator_set_burst_segment_count := Ttiepie_hw_generator_set_burst_segment_count( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_set_burst_segment_count' ) );
      tiepie_hw_generator_verify_burst_segment_count := Ttiepie_hw_generator_verify_burst_segment_count( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_burst_segment_count' ) );
      tiepie_hw_generator_verify_burst_segment_count_ex := Ttiepie_hw_generator_verify_burst_segment_count_ex( GetProcAddress( hLibTiePie , 'tiepie_hw_generator_verify_burst_segment_count_ex' ) );
      tiepie_hw_server_connect := Ttiepie_hw_server_connect( GetProcAddress( hLibTiePie , 'tiepie_hw_server_connect' ) );
      tiepie_hw_server_disconnect := Ttiepie_hw_server_disconnect( GetProcAddress( hLibTiePie , 'tiepie_hw_server_disconnect' ) );
      tiepie_hw_server_remove := Ttiepie_hw_server_remove( GetProcAddress( hLibTiePie , 'tiepie_hw_server_remove' ) );
      tiepie_hw_server_get_status := Ttiepie_hw_server_get_status( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_status' ) );
      tiepie_hw_server_get_last_error := Ttiepie_hw_server_get_last_error( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_last_error' ) );
      tiepie_hw_server_get_url := Ttiepie_hw_server_get_url( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_url' ) );
      tiepie_hw_server_get_id := Ttiepie_hw_server_get_id( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_id' ) );
      tiepie_hw_server_get_ip_address := Ttiepie_hw_server_get_ip_address( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_ip_address' ) );
      tiepie_hw_server_get_ip_port := Ttiepie_hw_server_get_ip_port( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_ip_port' ) );
      tiepie_hw_server_get_name := Ttiepie_hw_server_get_name( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_name' ) );
      tiepie_hw_server_get_description := Ttiepie_hw_server_get_description( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_description' ) );
      tiepie_hw_server_get_version := Ttiepie_hw_server_get_version( GetProcAddress( hLibTiePie , 'tiepie_hw_server_get_version' ) );
      tiepie_hw_pointerarray_new := Ttiepie_hw_pointerarray_new( GetProcAddress( hLibTiePie , 'tiepie_hw_pointerarray_new' ) );
      tiepie_hw_pointerarray_set := Ttiepie_hw_pointerarray_set( GetProcAddress( hLibTiePie , 'tiepie_hw_pointerarray_set' ) );
      tiepie_hw_pointerarray_delete := Ttiepie_hw_pointerarray_delete( GetProcAddress( hLibTiePie , 'tiepie_hw_pointerarray_delete' ) );

      tiepie_hw_init();
    end;
  end;
  Result := hLibTiePie <> 0;
end;

function tiepie_hw_unload : Boolean;
begin
  if hLibTiePie <> 0 then begin
    tiepie_hw_init := nil;
    tiepie_hw_is_initialized := nil;
    tiepie_hw_fini := nil;
    tiepie_hw_get_version := nil;
    tiepie_hw_get_config := nil;
    tiepie_hw_get_last_status := nil;
    tiepie_hw_get_last_status_str := nil;
    tiepie_hw_devicelist_update := nil;
    tiepie_hw_devicelist_get_usb_hotplug_detect_enabled := nil;
    tiepie_hw_devicelist_set_usb_hotplug_detect_enabled := nil;
    tiepie_hw_devicelist_get_count := nil;
    tiepie_hw_devicelist_get_item_by_product_id := nil;
    tiepie_hw_devicelist_get_item_by_index := nil;
    tiepie_hw_devicelist_get_item_by_serial_number := nil;
    tiepie_hw_devicelistitem_open_device := nil;
    tiepie_hw_devicelistitem_open_oscilloscope := nil;
    tiepie_hw_devicelistitem_open_generator := nil;
    tiepie_hw_devicelist_get_demo_device_info := nil;
    tiepie_hw_devicelist_create_demo_device := nil;
    tiepie_hw_devicelist_create_combined_device := nil;
    tiepie_hw_devicelist_create_and_open_combined_device := nil;
    tiepie_hw_devicelist_remove_device := nil;
    tiepie_hw_devicelist_remove_unused_devices := nil;
    tiepie_hw_devicelistitem_is_demo := nil;
    tiepie_hw_devicelistitem_can_open := nil;
    tiepie_hw_devicelistitem_opened_by := nil;
    tiepie_hw_devicelistitem_get_product_id := nil;
    tiepie_hw_devicelistitem_get_name := nil;
    tiepie_hw_devicelistitem_get_name_short := nil;
    tiepie_hw_devicelistitem_get_name_shortest := nil;
    tiepie_hw_devicelistitem_get_calibration_date := nil;
    tiepie_hw_devicelistitem_get_serial_number := nil;
    tiepie_hw_devicelistitem_get_ip_address := nil;
    tiepie_hw_devicelistitem_get_ip_port := nil;
    tiepie_hw_devicelistitem_has_server := nil;
    tiepie_hw_devicelistitem_get_server := nil;
    tiepie_hw_devicelistitem_get_types := nil;
    tiepie_hw_devicelistitem_get_contained_serial_numbers := nil;
    tiepie_hw_devicelistitemcombined_get_product_id := nil;
    tiepie_hw_devicelistitemcombined_get_name := nil;
    tiepie_hw_devicelistitemcombined_get_name_short := nil;
    tiepie_hw_devicelistitemcombined_get_name_shortest := nil;
    tiepie_hw_devicelistitemcombined_get_calibration_date := nil;
    tiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count := nil;
    tiepie_hw_devicelist_set_callback_device_added := nil;
    tiepie_hw_devicelist_set_callback_device_removed := nil;
    tiepie_hw_devicelist_set_callback_device_can_open_changed := nil;
    tiepie_hw_network_get_auto_detect_enabled := nil;
    tiepie_hw_network_set_auto_detect_enabled := nil;
    tiepie_hw_network_servers_add := nil;
    tiepie_hw_network_servers_remove := nil;
    tiepie_hw_network_servers_get_count := nil;
    tiepie_hw_network_servers_get_by_index := nil;
    tiepie_hw_network_servers_get_by_url := nil;
    tiepie_hw_network_servers_set_callback_added := nil;
    tiepie_hw_object_close := nil;
    tiepie_hw_object_is_removed := nil;
    tiepie_hw_object_get_interfaces := nil;
    tiepie_hw_object_set_event_callback := nil;
    tiepie_hw_device_get_calibration_date := nil;
    tiepie_hw_device_get_serial_number := nil;
    tiepie_hw_device_get_ip_address := nil;
    tiepie_hw_device_get_ip_port := nil;
    tiepie_hw_device_get_product_id := nil;
    tiepie_hw_device_get_type := nil;
    tiepie_hw_device_get_name := nil;
    tiepie_hw_device_get_name_short := nil;
    tiepie_hw_device_get_name_shortest := nil;
    tiepie_hw_device_has_wireless_trigger_module := nil;
    tiepie_hw_device_has_battery := nil;
    tiepie_hw_device_has_battery_ex := nil;
    tiepie_hw_device_get_battery_charge := nil;
    tiepie_hw_device_get_battery_charge_ex := nil;
    tiepie_hw_device_get_battery_time_to_empty := nil;
    tiepie_hw_device_get_battery_time_to_empty_ex := nil;
    tiepie_hw_device_get_battery_time_to_full := nil;
    tiepie_hw_device_get_battery_time_to_full_ex := nil;
    tiepie_hw_device_is_battery_charger_connected := nil;
    tiepie_hw_device_is_battery_charger_connected_ex := nil;
    tiepie_hw_device_is_battery_charging := nil;
    tiepie_hw_device_is_battery_charging_ex := nil;
    tiepie_hw_device_is_battery_broken := nil;
    tiepie_hw_device_is_battery_broken_ex := nil;
    tiepie_hw_device_trigger_get_input_count := nil;
    tiepie_hw_device_trigger_get_input_index_by_id := nil;
    tiepie_hw_oscilloscope_trigger_input_is_triggered := nil;
    tiepie_hw_device_trigger_input_get_enabled := nil;
    tiepie_hw_device_trigger_input_set_enabled := nil;
    tiepie_hw_device_trigger_input_get_kinds := nil;
    tiepie_hw_oscilloscope_trigger_input_get_kinds_ex := nil;
    tiepie_hw_device_trigger_input_get_kind := nil;
    tiepie_hw_device_trigger_input_set_kind := nil;
    tiepie_hw_device_trigger_input_is_available := nil;
    tiepie_hw_oscilloscope_trigger_input_is_available_ex := nil;
    tiepie_hw_device_trigger_input_get_id := nil;
    tiepie_hw_device_trigger_input_get_name := nil;
    tiepie_hw_device_trigger_get_output_count := nil;
    tiepie_hw_device_trigger_get_output_index_by_id := nil;
    tiepie_hw_device_trigger_output_get_enabled := nil;
    tiepie_hw_device_trigger_output_set_enabled := nil;
    tiepie_hw_device_trigger_output_get_events := nil;
    tiepie_hw_device_trigger_output_get_event := nil;
    tiepie_hw_device_trigger_output_set_event := nil;
    tiepie_hw_device_trigger_output_get_id := nil;
    tiepie_hw_device_trigger_output_get_name := nil;
    tiepie_hw_device_trigger_output_trigger := nil;
    tiepie_hw_oscilloscope_get_channel_count := nil;
    tiepie_hw_oscilloscope_channel_is_available := nil;
    tiepie_hw_oscilloscope_channel_is_available_ex := nil;
    tiepie_hw_oscilloscope_channel_get_connector_type := nil;
    tiepie_hw_oscilloscope_channel_is_differential := nil;
    tiepie_hw_oscilloscope_channel_is_isolated := nil;
    tiepie_hw_oscilloscope_channel_get_impedance := nil;
    tiepie_hw_oscilloscope_channel_get_bandwidths := nil;
    tiepie_hw_oscilloscope_channel_get_bandwidth := nil;
    tiepie_hw_oscilloscope_channel_set_bandwidth := nil;
    tiepie_hw_oscilloscope_channel_get_couplings := nil;
    tiepie_hw_oscilloscope_channel_get_coupling := nil;
    tiepie_hw_oscilloscope_channel_set_coupling := nil;
    tiepie_hw_oscilloscope_channel_get_enabled := nil;
    tiepie_hw_oscilloscope_channel_set_enabled := nil;
    tiepie_hw_oscilloscope_channel_get_auto_ranging := nil;
    tiepie_hw_oscilloscope_channel_set_auto_ranging := nil;
    tiepie_hw_oscilloscope_channel_get_ranges := nil;
    tiepie_hw_oscilloscope_channel_get_ranges_ex := nil;
    tiepie_hw_oscilloscope_channel_get_range := nil;
    tiepie_hw_oscilloscope_channel_set_range := nil;
    tiepie_hw_oscilloscope_channel_has_safeground := nil;
    tiepie_hw_oscilloscope_channel_get_safeground_enabled := nil;
    tiepie_hw_oscilloscope_channel_set_safeground_enabled := nil;
    tiepie_hw_oscilloscope_channel_get_safeground_threshold_min := nil;
    tiepie_hw_oscilloscope_channel_get_safeground_threshold_max := nil;
    tiepie_hw_oscilloscope_channel_get_safeground_threshold := nil;
    tiepie_hw_oscilloscope_channel_set_safeground_threshold := nil;
    tiepie_hw_oscilloscope_channel_verify_safeground_threshold := nil;
    tiepie_hw_oscilloscope_channel_has_trigger := nil;
    tiepie_hw_oscilloscope_channel_has_trigger_ex := nil;
    tiepie_hw_oscilloscope_channel_trigger_is_available := nil;
    tiepie_hw_oscilloscope_channel_trigger_is_available_ex := nil;
    tiepie_hw_oscilloscope_channel_trigger_is_triggered := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_enabled := nil;
    tiepie_hw_oscilloscope_channel_trigger_set_enabled := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_kinds := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_kinds_ex := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_kind := nil;
    tiepie_hw_oscilloscope_channel_trigger_set_kind := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_level_modes := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_level_mode := nil;
    tiepie_hw_oscilloscope_channel_trigger_set_level_mode := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_level_count := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_level := nil;
    tiepie_hw_oscilloscope_channel_trigger_set_level := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_hysteresis := nil;
    tiepie_hw_oscilloscope_channel_trigger_set_hysteresis := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_conditions := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_conditions_ex := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_condition := nil;
    tiepie_hw_oscilloscope_channel_trigger_set_condition := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_time_count := nil;
    tiepie_hw_oscilloscope_channel_trigger_get_time := nil;
    tiepie_hw_oscilloscope_channel_trigger_set_time := nil;
    tiepie_hw_oscilloscope_channel_trigger_verify_time := nil;
    tiepie_hw_oscilloscope_channel_trigger_verify_time_ex := nil;
    tiepie_hw_oscilloscope_is_demo := nil;
    tiepie_hw_oscilloscope_channel_demo_get_signal := nil;
    tiepie_hw_oscilloscope_channel_demo_set_signal := nil;
    tiepie_hw_oscilloscope_channel_demo_has_amplitude := nil;
    tiepie_hw_oscilloscope_channel_demo_get_amplitude := nil;
    tiepie_hw_oscilloscope_channel_demo_set_amplitude := nil;
    tiepie_hw_oscilloscope_channel_demo_has_frequency := nil;
    tiepie_hw_oscilloscope_channel_demo_get_frequency := nil;
    tiepie_hw_oscilloscope_channel_demo_set_frequency := nil;
    tiepie_hw_oscilloscope_channel_demo_has_offset := nil;
    tiepie_hw_oscilloscope_channel_demo_get_offset := nil;
    tiepie_hw_oscilloscope_channel_demo_set_offset := nil;
    tiepie_hw_oscilloscope_channel_demo_has_symmetry := nil;
    tiepie_hw_oscilloscope_channel_demo_get_symmetry := nil;
    tiepie_hw_oscilloscope_channel_demo_set_symmetry := nil;
    tiepie_hw_oscilloscope_channel_demo_has_phase := nil;
    tiepie_hw_oscilloscope_channel_demo_get_phase := nil;
    tiepie_hw_oscilloscope_channel_demo_set_phase := nil;
    tiepie_hw_oscilloscope_channel_demo_has_noise_enabled := nil;
    tiepie_hw_oscilloscope_channel_demo_get_noise_enabled := nil;
    tiepie_hw_oscilloscope_channel_demo_set_noise_enabled := nil;
    tiepie_hw_oscilloscope_get_data := nil;
    tiepie_hw_oscilloscope_get_data_1ch := nil;
    tiepie_hw_oscilloscope_get_data_2ch := nil;
    tiepie_hw_oscilloscope_get_data_3ch := nil;
    tiepie_hw_oscilloscope_get_data_4ch := nil;
    tiepie_hw_oscilloscope_get_data_5ch := nil;
    tiepie_hw_oscilloscope_get_data_6ch := nil;
    tiepie_hw_oscilloscope_get_data_7ch := nil;
    tiepie_hw_oscilloscope_get_data_8ch := nil;
    tiepie_hw_oscilloscope_get_valid_pre_sample_count := nil;
    tiepie_hw_oscilloscope_channel_get_data_value_range := nil;
    tiepie_hw_oscilloscope_channel_get_data_value_min := nil;
    tiepie_hw_oscilloscope_channel_get_data_value_max := nil;
    tiepie_hw_oscilloscope_get_data_raw := nil;
    tiepie_hw_oscilloscope_get_data_raw_1ch := nil;
    tiepie_hw_oscilloscope_get_data_raw_2ch := nil;
    tiepie_hw_oscilloscope_get_data_raw_3ch := nil;
    tiepie_hw_oscilloscope_get_data_raw_4ch := nil;
    tiepie_hw_oscilloscope_get_data_raw_5ch := nil;
    tiepie_hw_oscilloscope_get_data_raw_6ch := nil;
    tiepie_hw_oscilloscope_get_data_raw_7ch := nil;
    tiepie_hw_oscilloscope_get_data_raw_8ch := nil;
    tiepie_hw_oscilloscope_channel_get_data_raw_type := nil;
    tiepie_hw_oscilloscope_channel_get_data_raw_value_range := nil;
    tiepie_hw_oscilloscope_channel_get_data_raw_value_min := nil;
    tiepie_hw_oscilloscope_channel_get_data_raw_value_zero := nil;
    tiepie_hw_oscilloscope_channel_get_data_raw_value_max := nil;
    tiepie_hw_oscilloscope_channel_is_range_max_reachable := nil;
    tiepie_hw_oscilloscope_start := nil;
    tiepie_hw_oscilloscope_stop := nil;
    tiepie_hw_oscilloscope_force_trigger := nil;
    tiepie_hw_oscilloscope_get_measure_modes := nil;
    tiepie_hw_oscilloscope_get_measure_mode := nil;
    tiepie_hw_oscilloscope_set_measure_mode := nil;
    tiepie_hw_oscilloscope_is_running := nil;
    tiepie_hw_oscilloscope_is_triggered := nil;
    tiepie_hw_oscilloscope_is_timeout_triggered := nil;
    tiepie_hw_oscilloscope_is_force_triggered := nil;
    tiepie_hw_oscilloscope_is_data_ready := nil;
    tiepie_hw_oscilloscope_is_data_overflow := nil;
    tiepie_hw_oscilloscope_get_auto_resolution_modes := nil;
    tiepie_hw_oscilloscope_get_auto_resolution_mode := nil;
    tiepie_hw_oscilloscope_set_auto_resolution_mode := nil;
    tiepie_hw_oscilloscope_get_resolutions := nil;
    tiepie_hw_oscilloscope_get_resolution := nil;
    tiepie_hw_oscilloscope_set_resolution := nil;
    tiepie_hw_oscilloscope_is_resolution_enhanced := nil;
    tiepie_hw_oscilloscope_is_resolution_enhanced_ex := nil;
    tiepie_hw_oscilloscope_get_clock_sources := nil;
    tiepie_hw_oscilloscope_get_clock_source := nil;
    tiepie_hw_oscilloscope_set_clock_source := nil;
    tiepie_hw_oscilloscope_get_clock_source_frequencies := nil;
    tiepie_hw_oscilloscope_get_clock_source_frequencies_ex := nil;
    tiepie_hw_oscilloscope_get_clock_source_frequency := nil;
    tiepie_hw_oscilloscope_set_clock_source_frequency := nil;
    tiepie_hw_oscilloscope_get_clock_outputs := nil;
    tiepie_hw_oscilloscope_get_clock_output := nil;
    tiepie_hw_oscilloscope_set_clock_output := nil;
    tiepie_hw_oscilloscope_get_clock_output_frequencies := nil;
    tiepie_hw_oscilloscope_get_clock_output_frequencies_ex := nil;
    tiepie_hw_oscilloscope_get_clock_output_frequency := nil;
    tiepie_hw_oscilloscope_set_clock_output_frequency := nil;
    tiepie_hw_oscilloscope_get_sample_rate_max := nil;
    tiepie_hw_oscilloscope_get_sample_rate := nil;
    tiepie_hw_oscilloscope_set_sample_rate := nil;
    tiepie_hw_oscilloscope_verify_sample_rate := nil;
    tiepie_hw_oscilloscope_verify_sample_rate_ex := nil;
    tiepie_hw_oscilloscope_verify_sample_rates_ex := nil;
    tiepie_hw_oscilloscope_get_record_length_max := nil;
    tiepie_hw_oscilloscope_get_record_length_max_ex := nil;
    tiepie_hw_oscilloscope_get_record_length := nil;
    tiepie_hw_oscilloscope_set_record_length := nil;
    tiepie_hw_oscilloscope_verify_record_length := nil;
    tiepie_hw_oscilloscope_verify_record_length_ex := nil;
    tiepie_hw_oscilloscope_get_pre_sample_ratio := nil;
    tiepie_hw_oscilloscope_set_pre_sample_ratio := nil;
    tiepie_hw_oscilloscope_get_segment_count_max := nil;
    tiepie_hw_oscilloscope_get_segment_count_max_ex := nil;
    tiepie_hw_oscilloscope_get_segment_count := nil;
    tiepie_hw_oscilloscope_set_segment_count := nil;
    tiepie_hw_oscilloscope_verify_segment_count := nil;
    tiepie_hw_oscilloscope_verify_segment_count_ex := nil;
    tiepie_hw_oscilloscope_has_trigger := nil;
    tiepie_hw_oscilloscope_has_trigger_ex := nil;
    tiepie_hw_oscilloscope_trigger_get_timeout := nil;
    tiepie_hw_oscilloscope_trigger_set_timeout := nil;
    tiepie_hw_oscilloscope_trigger_verify_timeout := nil;
    tiepie_hw_oscilloscope_trigger_verify_timeout_ex := nil;
    tiepie_hw_oscilloscope_trigger_has_delay := nil;
    tiepie_hw_oscilloscope_trigger_has_delay_ex := nil;
    tiepie_hw_oscilloscope_trigger_get_delay_max := nil;
    tiepie_hw_oscilloscope_trigger_get_delay_max_ex := nil;
    tiepie_hw_oscilloscope_trigger_get_delay := nil;
    tiepie_hw_oscilloscope_trigger_set_delay := nil;
    tiepie_hw_oscilloscope_trigger_verify_delay := nil;
    tiepie_hw_oscilloscope_trigger_verify_delay_ex := nil;
    tiepie_hw_oscilloscope_has_presamples_valid := nil;
    tiepie_hw_oscilloscope_has_presamples_valid_ex := nil;
    tiepie_hw_oscilloscope_get_presamples_valid := nil;
    tiepie_hw_oscilloscope_set_presamples_valid := nil;
    tiepie_hw_oscilloscope_has_sureconnect := nil;
    tiepie_hw_oscilloscope_channel_has_sureconnect := nil;
    tiepie_hw_oscilloscope_start_sureconnect := nil;
    tiepie_hw_oscilloscope_start_sureconnect_ex := nil;
    tiepie_hw_oscilloscope_is_sureconnect_completed := nil;
    tiepie_hw_oscilloscope_get_sureconnect_data := nil;
    tiepie_hw_generator_get_connector_type := nil;
    tiepie_hw_generator_is_differential := nil;
    tiepie_hw_generator_get_impedance := nil;
    tiepie_hw_generator_get_resolution := nil;
    tiepie_hw_generator_get_output_value_min := nil;
    tiepie_hw_generator_get_output_value_max := nil;
    tiepie_hw_generator_get_output_value_min_max := nil;
    tiepie_hw_generator_is_controllable := nil;
    tiepie_hw_generator_is_running := nil;
    tiepie_hw_generator_get_status := nil;
    tiepie_hw_generator_get_output_enable := nil;
    tiepie_hw_generator_set_output_enable := nil;
    tiepie_hw_generator_has_output_invert := nil;
    tiepie_hw_generator_get_output_invert := nil;
    tiepie_hw_generator_set_output_invert := nil;
    tiepie_hw_generator_start := nil;
    tiepie_hw_generator_stop := nil;
    tiepie_hw_generator_get_signal_types := nil;
    tiepie_hw_generator_get_signal_type := nil;
    tiepie_hw_generator_set_signal_type := nil;
    tiepie_hw_generator_has_amplitude := nil;
    tiepie_hw_generator_has_amplitude_ex := nil;
    tiepie_hw_generator_get_amplitude_min := nil;
    tiepie_hw_generator_get_amplitude_max := nil;
    tiepie_hw_generator_get_amplitude_min_max_ex := nil;
    tiepie_hw_generator_get_amplitude := nil;
    tiepie_hw_generator_set_amplitude := nil;
    tiepie_hw_generator_verify_amplitude := nil;
    tiepie_hw_generator_verify_amplitude_ex := nil;
    tiepie_hw_generator_get_amplitude_ranges := nil;
    tiepie_hw_generator_get_amplitude_range := nil;
    tiepie_hw_generator_set_amplitude_range := nil;
    tiepie_hw_generator_get_amplitude_auto_ranging := nil;
    tiepie_hw_generator_set_amplitude_auto_ranging := nil;
    tiepie_hw_generator_has_offset := nil;
    tiepie_hw_generator_has_offset_ex := nil;
    tiepie_hw_generator_get_offset_min := nil;
    tiepie_hw_generator_get_offset_max := nil;
    tiepie_hw_generator_get_offset_min_max_ex := nil;
    tiepie_hw_generator_get_offset := nil;
    tiepie_hw_generator_set_offset := nil;
    tiepie_hw_generator_verify_offset := nil;
    tiepie_hw_generator_verify_offset_ex := nil;
    tiepie_hw_generator_get_frequency_modes := nil;
    tiepie_hw_generator_get_frequency_modes_ex := nil;
    tiepie_hw_generator_get_frequency_mode := nil;
    tiepie_hw_generator_set_frequency_mode := nil;
    tiepie_hw_generator_has_frequency := nil;
    tiepie_hw_generator_has_frequency_ex := nil;
    tiepie_hw_generator_get_frequency_min := nil;
    tiepie_hw_generator_get_frequency_max := nil;
    tiepie_hw_generator_get_frequency_min_max := nil;
    tiepie_hw_generator_get_frequency_min_max_ex := nil;
    tiepie_hw_generator_get_frequency := nil;
    tiepie_hw_generator_set_frequency := nil;
    tiepie_hw_generator_verify_frequency := nil;
    tiepie_hw_generator_verify_frequency_ex := nil;
    tiepie_hw_generator_has_phase := nil;
    tiepie_hw_generator_has_phase_ex := nil;
    tiepie_hw_generator_get_phase_min := nil;
    tiepie_hw_generator_get_phase_max := nil;
    tiepie_hw_generator_get_phase_min_max_ex := nil;
    tiepie_hw_generator_get_phase := nil;
    tiepie_hw_generator_set_phase := nil;
    tiepie_hw_generator_verify_phase := nil;
    tiepie_hw_generator_verify_phase_ex := nil;
    tiepie_hw_generator_has_symmetry := nil;
    tiepie_hw_generator_has_symmetry_ex := nil;
    tiepie_hw_generator_get_symmetry_min := nil;
    tiepie_hw_generator_get_symmetry_max := nil;
    tiepie_hw_generator_get_symmetry_min_max_ex := nil;
    tiepie_hw_generator_get_symmetry := nil;
    tiepie_hw_generator_set_symmetry := nil;
    tiepie_hw_generator_verify_symmetry := nil;
    tiepie_hw_generator_verify_symmetry_ex := nil;
    tiepie_hw_generator_has_width := nil;
    tiepie_hw_generator_has_width_ex := nil;
    tiepie_hw_generator_get_width_min := nil;
    tiepie_hw_generator_get_width_max := nil;
    tiepie_hw_generator_get_width_min_max_ex := nil;
    tiepie_hw_generator_get_width := nil;
    tiepie_hw_generator_set_width := nil;
    tiepie_hw_generator_verify_width := nil;
    tiepie_hw_generator_verify_width_ex := nil;
    tiepie_hw_generator_has_edge_time := nil;
    tiepie_hw_generator_has_edge_time_ex := nil;
    tiepie_hw_generator_get_leading_edge_time_min := nil;
    tiepie_hw_generator_get_leading_edge_time_max := nil;
    tiepie_hw_generator_get_leading_edge_time_min_max_ex := nil;
    tiepie_hw_generator_get_leading_edge_time := nil;
    tiepie_hw_generator_set_leading_edge_time := nil;
    tiepie_hw_generator_verify_leading_edge_time := nil;
    tiepie_hw_generator_verify_leading_edge_time_ex := nil;
    tiepie_hw_generator_get_trailing_edge_time_min := nil;
    tiepie_hw_generator_get_trailing_edge_time_max := nil;
    tiepie_hw_generator_get_trailing_edge_time_min_max_ex := nil;
    tiepie_hw_generator_get_trailing_edge_time := nil;
    tiepie_hw_generator_set_trailing_edge_time := nil;
    tiepie_hw_generator_verify_trailing_edge_time := nil;
    tiepie_hw_generator_verify_trailing_edge_time_ex := nil;
    tiepie_hw_generator_has_data := nil;
    tiepie_hw_generator_has_data_ex := nil;
    tiepie_hw_generator_get_data_length_min := nil;
    tiepie_hw_generator_get_data_length_max := nil;
    tiepie_hw_generator_get_data_length_min_max_ex := nil;
    tiepie_hw_generator_get_data_length := nil;
    tiepie_hw_generator_verify_data_length := nil;
    tiepie_hw_generator_verify_data_length_ex := nil;
    tiepie_hw_generator_set_data := nil;
    tiepie_hw_generator_set_data_ex := nil;
    tiepie_hw_generator_get_data_raw_type := nil;
    tiepie_hw_generator_get_data_raw_value_range := nil;
    tiepie_hw_generator_get_data_raw_value_min := nil;
    tiepie_hw_generator_get_data_raw_value_zero := nil;
    tiepie_hw_generator_get_data_raw_value_max := nil;
    tiepie_hw_generator_set_data_raw := nil;
    tiepie_hw_generator_set_data_raw_ex := nil;
    tiepie_hw_generator_get_modes := nil;
    tiepie_hw_generator_get_modes_ex := nil;
    tiepie_hw_generator_get_modes_native := nil;
    tiepie_hw_generator_get_mode := nil;
    tiepie_hw_generator_set_mode := nil;
    tiepie_hw_generator_is_burst_active := nil;
    tiepie_hw_generator_get_burst_count_min := nil;
    tiepie_hw_generator_get_burst_count_max := nil;
    tiepie_hw_generator_get_burst_count_min_max_ex := nil;
    tiepie_hw_generator_get_burst_count := nil;
    tiepie_hw_generator_set_burst_count := nil;
    tiepie_hw_generator_get_burst_sample_count_min := nil;
    tiepie_hw_generator_get_burst_sample_count_max := nil;
    tiepie_hw_generator_get_burst_sample_count_min_max_ex := nil;
    tiepie_hw_generator_get_burst_sample_count := nil;
    tiepie_hw_generator_set_burst_sample_count := nil;
    tiepie_hw_generator_get_burst_segment_count_min := nil;
    tiepie_hw_generator_get_burst_segment_count_max := nil;
    tiepie_hw_generator_get_burst_segment_count_min_max_ex := nil;
    tiepie_hw_generator_get_burst_segment_count := nil;
    tiepie_hw_generator_set_burst_segment_count := nil;
    tiepie_hw_generator_verify_burst_segment_count := nil;
    tiepie_hw_generator_verify_burst_segment_count_ex := nil;
    tiepie_hw_server_connect := nil;
    tiepie_hw_server_disconnect := nil;
    tiepie_hw_server_remove := nil;
    tiepie_hw_server_get_status := nil;
    tiepie_hw_server_get_last_error := nil;
    tiepie_hw_server_get_url := nil;
    tiepie_hw_server_get_id := nil;
    tiepie_hw_server_get_ip_address := nil;
    tiepie_hw_server_get_ip_port := nil;
    tiepie_hw_server_get_name := nil;
    tiepie_hw_server_get_description := nil;
    tiepie_hw_server_get_version := nil;
    tiepie_hw_pointerarray_new := nil;
    tiepie_hw_pointerarray_set := nil;
    tiepie_hw_pointerarray_delete := nil;

    tiepie_hw_exit();

    FreeLibrary(hLibTiePie);

    hLibTiePie := 0;
  end;
  Result := hLibTiePie = 0;
end;

{$else} // LIBTIEPIE_STATIC

procedure tiepie_hw_init; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_init'{$endif};
function tiepie_hw_is_initialized : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_is_initialized'{$endif};
procedure tiepie_hw_fini; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_fini'{$endif};
function tiepie_hw_get_version : Ptiepie_hw_version; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_get_version'{$endif};
function tiepie_hw_get_config(buffer : PByte; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_get_config'{$endif};
function tiepie_hw_get_last_status : Ttiepie_hw_status; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_get_last_status'{$endif};
function tiepie_hw_get_last_status_str : PAnsiChar; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_get_last_status_str'{$endif};
procedure tiepie_hw_devicelist_update; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_update'{$endif};
function tiepie_hw_devicelist_get_usb_hotplug_detect_enabled : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_get_usb_hotplug_detect_enabled'{$endif};
function tiepie_hw_devicelist_set_usb_hotplug_detect_enabled(value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_set_usb_hotplug_detect_enabled'{$endif};
function tiepie_hw_devicelist_get_count : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_get_count'{$endif};
function tiepie_hw_devicelist_get_item_by_product_id(product_id : Ttiepie_hw_productid) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_get_item_by_product_id'{$endif};
function tiepie_hw_devicelist_get_item_by_index(index : LongWord) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_get_item_by_index'{$endif};
function tiepie_hw_devicelist_get_item_by_serial_number(serial_number : LongWord) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_get_item_by_serial_number'{$endif};
function tiepie_hw_devicelistitem_open_device(handle : Ttiepie_hw_handle; device_type : LongWord) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_open_device'{$endif};
function tiepie_hw_devicelistitem_open_oscilloscope(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_open_oscilloscope'{$endif};
function tiepie_hw_devicelistitem_open_generator(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_open_generator'{$endif};
function tiepie_hw_devicelist_get_demo_device_info : Ptiepie_hw_demo_info; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_get_demo_device_info'{$endif};
function tiepie_hw_devicelist_create_demo_device(product_id : Ttiepie_hw_productid) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_create_demo_device'{$endif};
function tiepie_hw_devicelist_create_combined_device(handles : Ptiepie_hw_handle; count : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_create_combined_device'{$endif};
function tiepie_hw_devicelist_create_and_open_combined_device(handles : Ptiepie_hw_handle; count : LongWord) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_create_and_open_combined_device'{$endif};
procedure tiepie_hw_devicelist_remove_device(serial_number : LongWord; force : ByteBool); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_remove_device'{$endif};
procedure tiepie_hw_devicelist_remove_unused_devices; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_remove_unused_devices'{$endif};
function tiepie_hw_devicelistitem_is_demo(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_is_demo'{$endif};
function tiepie_hw_devicelistitem_can_open(handle : Ttiepie_hw_handle; device_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_can_open'{$endif};
function tiepie_hw_devicelistitem_opened_by(handle : Ttiepie_hw_handle; device_type : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_opened_by'{$endif};
function tiepie_hw_devicelistitem_get_product_id(handle : Ttiepie_hw_handle) : Ttiepie_hw_productid; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_product_id'{$endif};
function tiepie_hw_devicelistitem_get_name(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_name'{$endif};
function tiepie_hw_devicelistitem_get_name_short(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_name_short'{$endif};
function tiepie_hw_devicelistitem_get_name_shortest(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_name_shortest'{$endif};
function tiepie_hw_devicelistitem_get_calibration_date(handle : Ttiepie_hw_handle) : Ttiepie_hw_date; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_calibration_date'{$endif};
function tiepie_hw_devicelistitem_get_serial_number(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_serial_number'{$endif};
function tiepie_hw_devicelistitem_get_ip_address(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_ip_address'{$endif};
function tiepie_hw_devicelistitem_get_ip_port(handle : Ttiepie_hw_handle) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_ip_port'{$endif};
function tiepie_hw_devicelistitem_has_server(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_has_server'{$endif};
function tiepie_hw_devicelistitem_get_server(handle : Ttiepie_hw_handle) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_server'{$endif};
function tiepie_hw_devicelistitem_get_types(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_types'{$endif};
function tiepie_hw_devicelistitem_get_contained_serial_numbers(handle : Ttiepie_hw_handle; buffer : PLongWord; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitem_get_contained_serial_numbers'{$endif};
function tiepie_hw_devicelistitemcombined_get_product_id(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Ttiepie_hw_productid; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitemcombined_get_product_id'{$endif};
function tiepie_hw_devicelistitemcombined_get_name(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitemcombined_get_name'{$endif};
function tiepie_hw_devicelistitemcombined_get_name_short(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitemcombined_get_name_short'{$endif};
function tiepie_hw_devicelistitemcombined_get_name_shortest(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitemcombined_get_name_shortest'{$endif};
function tiepie_hw_devicelistitemcombined_get_calibration_date(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Ttiepie_hw_date; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitemcombined_get_calibration_date'{$endif};
function tiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelistitemcombined_get_oscilloscope_channel_count'{$endif};
procedure tiepie_hw_devicelist_set_callback_device_added(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_set_callback_device_added'{$endif};
procedure tiepie_hw_devicelist_set_callback_device_removed(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_set_callback_device_removed'{$endif};
procedure tiepie_hw_devicelist_set_callback_device_can_open_changed(callback : Ttiepie_hw_devicelist_callback; data : Pointer); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_devicelist_set_callback_device_can_open_changed'{$endif};
function tiepie_hw_network_get_auto_detect_enabled : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_network_get_auto_detect_enabled'{$endif};
function tiepie_hw_network_set_auto_detect_enabled(value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_network_set_auto_detect_enabled'{$endif};
function tiepie_hw_network_servers_add(url : PAnsiChar; length : LongWord; handle : Ptiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_network_servers_add'{$endif};
function tiepie_hw_network_servers_remove(url : PAnsiChar; length : LongWord; force : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_network_servers_remove'{$endif};
function tiepie_hw_network_servers_get_count : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_network_servers_get_count'{$endif};
function tiepie_hw_network_servers_get_by_index(index : LongWord) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_network_servers_get_by_index'{$endif};
function tiepie_hw_network_servers_get_by_url(url : PAnsiChar; length : LongWord) : Ttiepie_hw_handle; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_network_servers_get_by_url'{$endif};
procedure tiepie_hw_network_servers_set_callback_added(callback : Ttiepie_hw_handle_callback; data : Pointer); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_network_servers_set_callback_added'{$endif};
procedure tiepie_hw_object_close(handle : Ttiepie_hw_handle); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_object_close'{$endif};
function tiepie_hw_object_is_removed(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_object_is_removed'{$endif};
function tiepie_hw_object_get_interfaces(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_object_get_interfaces'{$endif};
procedure tiepie_hw_object_set_event_callback(handle : Ttiepie_hw_handle; callback : Ttiepie_hw_event_callback; data : Pointer); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_object_set_event_callback'{$endif};
function tiepie_hw_device_get_calibration_date(handle : Ttiepie_hw_handle) : Ttiepie_hw_date; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_calibration_date'{$endif};
function tiepie_hw_device_get_serial_number(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_serial_number'{$endif};
function tiepie_hw_device_get_ip_address(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_ip_address'{$endif};
function tiepie_hw_device_get_ip_port(handle : Ttiepie_hw_handle) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_ip_port'{$endif};
function tiepie_hw_device_get_product_id(handle : Ttiepie_hw_handle) : Ttiepie_hw_productid; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_product_id'{$endif};
function tiepie_hw_device_get_type(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_type'{$endif};
function tiepie_hw_device_get_name(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_name'{$endif};
function tiepie_hw_device_get_name_short(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_name_short'{$endif};
function tiepie_hw_device_get_name_shortest(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_name_shortest'{$endif};
function tiepie_hw_device_has_wireless_trigger_module(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_has_wireless_trigger_module'{$endif};
function tiepie_hw_device_has_battery(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_has_battery'{$endif};
function tiepie_hw_device_has_battery_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_has_battery_ex'{$endif};
function tiepie_hw_device_get_battery_charge(handle : Ttiepie_hw_handle) : ShortInt; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_battery_charge'{$endif};
function tiepie_hw_device_get_battery_charge_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ShortInt; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_battery_charge_ex'{$endif};
function tiepie_hw_device_get_battery_time_to_empty(handle : Ttiepie_hw_handle) : LongInt; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_battery_time_to_empty'{$endif};
function tiepie_hw_device_get_battery_time_to_empty_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : LongInt; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_battery_time_to_empty_ex'{$endif};
function tiepie_hw_device_get_battery_time_to_full(handle : Ttiepie_hw_handle) : LongInt; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_battery_time_to_full'{$endif};
function tiepie_hw_device_get_battery_time_to_full_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : LongInt; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_get_battery_time_to_full_ex'{$endif};
function tiepie_hw_device_is_battery_charger_connected(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_is_battery_charger_connected'{$endif};
function tiepie_hw_device_is_battery_charger_connected_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_is_battery_charger_connected_ex'{$endif};
function tiepie_hw_device_is_battery_charging(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_is_battery_charging'{$endif};
function tiepie_hw_device_is_battery_charging_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_is_battery_charging_ex'{$endif};
function tiepie_hw_device_is_battery_broken(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_is_battery_broken'{$endif};
function tiepie_hw_device_is_battery_broken_ex(handle : Ttiepie_hw_handle; serial_number : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_is_battery_broken_ex'{$endif};
function tiepie_hw_device_trigger_get_input_count(handle : Ttiepie_hw_handle) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_get_input_count'{$endif};
function tiepie_hw_device_trigger_get_input_index_by_id(handle : Ttiepie_hw_handle; id : LongWord) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_get_input_index_by_id'{$endif};
function tiepie_hw_oscilloscope_trigger_input_is_triggered(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_input_is_triggered'{$endif};
function tiepie_hw_device_trigger_input_get_enabled(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_input_get_enabled'{$endif};
function tiepie_hw_device_trigger_input_set_enabled(handle : Ttiepie_hw_handle; input : Word; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_input_set_enabled'{$endif};
function tiepie_hw_device_trigger_input_get_kinds(handle : Ttiepie_hw_handle; input : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_input_get_kinds'{$endif};
function tiepie_hw_oscilloscope_trigger_input_get_kinds_ex(handle : Ttiepie_hw_handle; input : Word; measure_mode : LongWord) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_input_get_kinds_ex'{$endif};
function tiepie_hw_device_trigger_input_get_kind(handle : Ttiepie_hw_handle; input : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_input_get_kind'{$endif};
function tiepie_hw_device_trigger_input_set_kind(handle : Ttiepie_hw_handle; input : Word; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_input_set_kind'{$endif};
function tiepie_hw_device_trigger_input_is_available(handle : Ttiepie_hw_handle; input : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_input_is_available'{$endif};
function tiepie_hw_oscilloscope_trigger_input_is_available_ex(handle : Ttiepie_hw_handle; input : Word; measure_mode : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_input_is_available_ex'{$endif};
function tiepie_hw_device_trigger_input_get_id(handle : Ttiepie_hw_handle; input : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_input_get_id'{$endif};
function tiepie_hw_device_trigger_input_get_name(handle : Ttiepie_hw_handle; input : Word; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_input_get_name'{$endif};
function tiepie_hw_device_trigger_get_output_count(handle : Ttiepie_hw_handle) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_get_output_count'{$endif};
function tiepie_hw_device_trigger_get_output_index_by_id(handle : Ttiepie_hw_handle; id : LongWord) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_get_output_index_by_id'{$endif};
function tiepie_hw_device_trigger_output_get_enabled(handle : Ttiepie_hw_handle; output : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_output_get_enabled'{$endif};
function tiepie_hw_device_trigger_output_set_enabled(handle : Ttiepie_hw_handle; output : Word; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_output_set_enabled'{$endif};
function tiepie_hw_device_trigger_output_get_events(handle : Ttiepie_hw_handle; output : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_output_get_events'{$endif};
function tiepie_hw_device_trigger_output_get_event(handle : Ttiepie_hw_handle; output : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_output_get_event'{$endif};
function tiepie_hw_device_trigger_output_set_event(handle : Ttiepie_hw_handle; output : Word; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_output_set_event'{$endif};
function tiepie_hw_device_trigger_output_get_id(handle : Ttiepie_hw_handle; output : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_output_get_id'{$endif};
function tiepie_hw_device_trigger_output_get_name(handle : Ttiepie_hw_handle; output : Word; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_output_get_name'{$endif};
function tiepie_hw_device_trigger_output_trigger(handle : Ttiepie_hw_handle; output : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_device_trigger_output_trigger'{$endif};
function tiepie_hw_oscilloscope_get_channel_count(handle : Ttiepie_hw_handle) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_channel_count'{$endif};
function tiepie_hw_oscilloscope_channel_is_available(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_is_available'{$endif};
function tiepie_hw_oscilloscope_channel_is_available_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; sample_rate : Double; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_is_available_ex'{$endif};
function tiepie_hw_oscilloscope_channel_get_connector_type(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_connector_type'{$endif};
function tiepie_hw_oscilloscope_channel_is_differential(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_is_differential'{$endif};
function tiepie_hw_oscilloscope_channel_is_isolated(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_is_isolated'{$endif};
function tiepie_hw_oscilloscope_channel_get_impedance(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_impedance'{$endif};
function tiepie_hw_oscilloscope_channel_get_bandwidths(handle : Ttiepie_hw_handle; ch : Word; list : PDouble; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_bandwidths'{$endif};
function tiepie_hw_oscilloscope_channel_get_bandwidth(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_bandwidth'{$endif};
function tiepie_hw_oscilloscope_channel_set_bandwidth(handle : Ttiepie_hw_handle; ch : Word; bandwidth : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_set_bandwidth'{$endif};
function tiepie_hw_oscilloscope_channel_get_couplings(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_couplings'{$endif};
function tiepie_hw_oscilloscope_channel_get_coupling(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_coupling'{$endif};
function tiepie_hw_oscilloscope_channel_set_coupling(handle : Ttiepie_hw_handle; ch : Word; coupling : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_set_coupling'{$endif};
function tiepie_hw_oscilloscope_channel_get_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_enabled'{$endif};
function tiepie_hw_oscilloscope_channel_set_enabled(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_set_enabled'{$endif};
function tiepie_hw_oscilloscope_channel_get_auto_ranging(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_auto_ranging'{$endif};
function tiepie_hw_oscilloscope_channel_set_auto_ranging(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_set_auto_ranging'{$endif};
function tiepie_hw_oscilloscope_channel_get_ranges(handle : Ttiepie_hw_handle; ch : Word; list : PDouble; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_ranges'{$endif};
function tiepie_hw_oscilloscope_channel_get_ranges_ex(handle : Ttiepie_hw_handle; ch : Word; coupling : UInt64; list : PDouble; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_ranges_ex'{$endif};
function tiepie_hw_oscilloscope_channel_get_range(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_range'{$endif};
function tiepie_hw_oscilloscope_channel_set_range(handle : Ttiepie_hw_handle; ch : Word; range : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_set_range'{$endif};
function tiepie_hw_oscilloscope_channel_has_safeground(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_has_safeground'{$endif};
function tiepie_hw_oscilloscope_channel_get_safeground_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_safeground_enabled'{$endif};
function tiepie_hw_oscilloscope_channel_set_safeground_enabled(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_set_safeground_enabled'{$endif};
function tiepie_hw_oscilloscope_channel_get_safeground_threshold_min(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_safeground_threshold_min'{$endif};
function tiepie_hw_oscilloscope_channel_get_safeground_threshold_max(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_safeground_threshold_max'{$endif};
function tiepie_hw_oscilloscope_channel_get_safeground_threshold(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_safeground_threshold'{$endif};
function tiepie_hw_oscilloscope_channel_set_safeground_threshold(handle : Ttiepie_hw_handle; ch : Word; threshold : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_set_safeground_threshold'{$endif};
function tiepie_hw_oscilloscope_channel_verify_safeground_threshold(handle : Ttiepie_hw_handle; ch : Word; threshold : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_verify_safeground_threshold'{$endif};
function tiepie_hw_oscilloscope_channel_has_trigger(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_has_trigger'{$endif};
function tiepie_hw_oscilloscope_channel_has_trigger_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_has_trigger_ex'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_is_available(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_is_available'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_is_available_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; sample_rate : Double; resolution : Byte; channel_enabled, channel_trigger_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_is_available_ex'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_is_triggered(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_is_triggered'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_enabled'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_set_enabled(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_set_enabled'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_kinds(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_kinds'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_kinds_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_kinds_ex'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_kind(handle : Ttiepie_hw_handle; ch : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_kind'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_set_kind(handle : Ttiepie_hw_handle; ch : Word; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_set_kind'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_level_modes(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_level_modes'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_level_mode(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_level_mode'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_set_level_mode(handle : Ttiepie_hw_handle; ch : Word; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_set_level_mode'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_level_count(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_level_count'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_level(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_level'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_set_level(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_set_level'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_hysteresis_count'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_hysteresis(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_hysteresis'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_set_hysteresis(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_set_hysteresis'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_conditions(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_conditions'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_conditions_ex(handle : Ttiepie_hw_handle; ch : Word; measure_mode : LongWord; trigger_kind : UInt64) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_conditions_ex'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_condition(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_condition'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_set_condition(handle : Ttiepie_hw_handle; ch : Word; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_set_condition'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_time_count(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_time_count'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_get_time(handle : Ttiepie_hw_handle; ch : Word; index : LongWord) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_get_time'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_set_time(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_set_time'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_verify_time(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_verify_time'{$endif};
function tiepie_hw_oscilloscope_channel_trigger_verify_time_ex(handle : Ttiepie_hw_handle; ch : Word; index : LongWord; value : Double; measure_mode : LongWord; sample_rate : Double; trigger_kind : UInt64; trigger_condition : LongWord) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_trigger_verify_time_ex'{$endif};
function tiepie_hw_oscilloscope_is_demo(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_demo'{$endif};
function tiepie_hw_oscilloscope_channel_demo_get_signal(handle : Ttiepie_hw_handle; ch : Word) : Ttiepie_hw_demosignal; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_get_signal'{$endif};
function tiepie_hw_oscilloscope_channel_demo_set_signal(handle : Ttiepie_hw_handle; ch : Word; value : Ttiepie_hw_demosignal) : Ttiepie_hw_demosignal; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_set_signal'{$endif};
function tiepie_hw_oscilloscope_channel_demo_has_amplitude(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_has_amplitude'{$endif};
function tiepie_hw_oscilloscope_channel_demo_get_amplitude(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_get_amplitude'{$endif};
function tiepie_hw_oscilloscope_channel_demo_set_amplitude(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_set_amplitude'{$endif};
function tiepie_hw_oscilloscope_channel_demo_has_frequency(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_has_frequency'{$endif};
function tiepie_hw_oscilloscope_channel_demo_get_frequency(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_get_frequency'{$endif};
function tiepie_hw_oscilloscope_channel_demo_set_frequency(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_set_frequency'{$endif};
function tiepie_hw_oscilloscope_channel_demo_has_offset(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_has_offset'{$endif};
function tiepie_hw_oscilloscope_channel_demo_get_offset(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_get_offset'{$endif};
function tiepie_hw_oscilloscope_channel_demo_set_offset(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_set_offset'{$endif};
function tiepie_hw_oscilloscope_channel_demo_has_symmetry(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_has_symmetry'{$endif};
function tiepie_hw_oscilloscope_channel_demo_get_symmetry(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_get_symmetry'{$endif};
function tiepie_hw_oscilloscope_channel_demo_set_symmetry(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_set_symmetry'{$endif};
function tiepie_hw_oscilloscope_channel_demo_has_phase(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_has_phase'{$endif};
function tiepie_hw_oscilloscope_channel_demo_get_phase(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_get_phase'{$endif};
function tiepie_hw_oscilloscope_channel_demo_set_phase(handle : Ttiepie_hw_handle; ch : Word; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_set_phase'{$endif};
function tiepie_hw_oscilloscope_channel_demo_has_noise_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_has_noise_enabled'{$endif};
function tiepie_hw_oscilloscope_channel_demo_get_noise_enabled(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_get_noise_enabled'{$endif};
function tiepie_hw_oscilloscope_channel_demo_set_noise_enabled(handle : Ttiepie_hw_handle; ch : Word; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_demo_set_noise_enabled'{$endif};
function tiepie_hw_oscilloscope_get_data(handle : Ttiepie_hw_handle; buffers : PPSingle; channel_count : Word; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data'{$endif};
function tiepie_hw_oscilloscope_get_data_1ch(handle : Ttiepie_hw_handle; buffer_ch1 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_1ch'{$endif};
function tiepie_hw_oscilloscope_get_data_2ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_2ch'{$endif};
function tiepie_hw_oscilloscope_get_data_3ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_3ch'{$endif};
function tiepie_hw_oscilloscope_get_data_4ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_4ch'{$endif};
function tiepie_hw_oscilloscope_get_data_5ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_5ch'{$endif};
function tiepie_hw_oscilloscope_get_data_6ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_6ch'{$endif};
function tiepie_hw_oscilloscope_get_data_7ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6, buffer_ch7 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_7ch'{$endif};
function tiepie_hw_oscilloscope_get_data_8ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6, buffer_ch7, buffer_ch8 : PSingle; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_8ch'{$endif};
function tiepie_hw_oscilloscope_get_valid_pre_sample_count(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_valid_pre_sample_count'{$endif};
procedure tiepie_hw_oscilloscope_channel_get_data_value_range(handle : Ttiepie_hw_handle; ch : Word; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_data_value_range'{$endif};
function tiepie_hw_oscilloscope_channel_get_data_value_min(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_data_value_min'{$endif};
function tiepie_hw_oscilloscope_channel_get_data_value_max(handle : Ttiepie_hw_handle; ch : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_data_value_max'{$endif};
function tiepie_hw_oscilloscope_get_data_raw(handle : Ttiepie_hw_handle; buffers : PPointer; channel_count : Word; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw'{$endif};
function tiepie_hw_oscilloscope_get_data_raw_1ch(handle : Ttiepie_hw_handle; buffer_ch1 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw_1ch'{$endif};
function tiepie_hw_oscilloscope_get_data_raw_2ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw_2ch'{$endif};
function tiepie_hw_oscilloscope_get_data_raw_3ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw_3ch'{$endif};
function tiepie_hw_oscilloscope_get_data_raw_4ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw_4ch'{$endif};
function tiepie_hw_oscilloscope_get_data_raw_5ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw_5ch'{$endif};
function tiepie_hw_oscilloscope_get_data_raw_6ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw_6ch'{$endif};
function tiepie_hw_oscilloscope_get_data_raw_7ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6, buffer_ch7 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw_7ch'{$endif};
function tiepie_hw_oscilloscope_get_data_raw_8ch(handle : Ttiepie_hw_handle; buffer_ch1, buffer_ch2, buffer_ch3, buffer_ch4, buffer_ch5, buffer_ch6, buffer_ch7, buffer_ch8 : Pointer; start_index, sample_count : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_data_raw_8ch'{$endif};
function tiepie_hw_oscilloscope_channel_get_data_raw_type(handle : Ttiepie_hw_handle; ch : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_data_raw_type'{$endif};
procedure tiepie_hw_oscilloscope_channel_get_data_raw_value_range(handle : Ttiepie_hw_handle; ch : Word; min, zero, max : PInt64); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_data_raw_value_range'{$endif};
function tiepie_hw_oscilloscope_channel_get_data_raw_value_min(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_data_raw_value_min'{$endif};
function tiepie_hw_oscilloscope_channel_get_data_raw_value_zero(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_data_raw_value_zero'{$endif};
function tiepie_hw_oscilloscope_channel_get_data_raw_value_max(handle : Ttiepie_hw_handle; ch : Word) : Int64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_get_data_raw_value_max'{$endif};
function tiepie_hw_oscilloscope_channel_is_range_max_reachable(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_is_range_max_reachable'{$endif};
function tiepie_hw_oscilloscope_start(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_start'{$endif};
function tiepie_hw_oscilloscope_stop(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_stop'{$endif};
function tiepie_hw_oscilloscope_force_trigger(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_force_trigger'{$endif};
function tiepie_hw_oscilloscope_get_measure_modes(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_measure_modes'{$endif};
function tiepie_hw_oscilloscope_get_measure_mode(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_measure_mode'{$endif};
function tiepie_hw_oscilloscope_set_measure_mode(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_measure_mode'{$endif};
function tiepie_hw_oscilloscope_is_running(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_running'{$endif};
function tiepie_hw_oscilloscope_is_triggered(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_triggered'{$endif};
function tiepie_hw_oscilloscope_is_timeout_triggered(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_timeout_triggered'{$endif};
function tiepie_hw_oscilloscope_is_force_triggered(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_force_triggered'{$endif};
function tiepie_hw_oscilloscope_is_data_ready(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_data_ready'{$endif};
function tiepie_hw_oscilloscope_is_data_overflow(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_data_overflow'{$endif};
function tiepie_hw_oscilloscope_get_auto_resolution_modes(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_auto_resolution_modes'{$endif};
function tiepie_hw_oscilloscope_get_auto_resolution_mode(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_auto_resolution_mode'{$endif};
function tiepie_hw_oscilloscope_set_auto_resolution_mode(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_auto_resolution_mode'{$endif};
function tiepie_hw_oscilloscope_get_resolutions(handle : Ttiepie_hw_handle; list : PByte; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_resolutions'{$endif};
function tiepie_hw_oscilloscope_get_resolution(handle : Ttiepie_hw_handle) : Byte; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_resolution'{$endif};
function tiepie_hw_oscilloscope_set_resolution(handle : Ttiepie_hw_handle; value : Byte) : Byte; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_resolution'{$endif};
function tiepie_hw_oscilloscope_is_resolution_enhanced(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_resolution_enhanced'{$endif};
function tiepie_hw_oscilloscope_is_resolution_enhanced_ex(handle : Ttiepie_hw_handle; value : Byte) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_resolution_enhanced_ex'{$endif};
function tiepie_hw_oscilloscope_get_clock_sources(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_sources'{$endif};
function tiepie_hw_oscilloscope_get_clock_source(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_source'{$endif};
function tiepie_hw_oscilloscope_set_clock_source(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_clock_source'{$endif};
function tiepie_hw_oscilloscope_get_clock_source_frequencies(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_source_frequencies'{$endif};
function tiepie_hw_oscilloscope_get_clock_source_frequencies_ex(handle : Ttiepie_hw_handle; value : LongWord; list : PDouble; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_source_frequencies_ex'{$endif};
function tiepie_hw_oscilloscope_get_clock_source_frequency(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_source_frequency'{$endif};
function tiepie_hw_oscilloscope_set_clock_source_frequency(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_clock_source_frequency'{$endif};
function tiepie_hw_oscilloscope_get_clock_outputs(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_outputs'{$endif};
function tiepie_hw_oscilloscope_get_clock_output(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_output'{$endif};
function tiepie_hw_oscilloscope_set_clock_output(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_clock_output'{$endif};
function tiepie_hw_oscilloscope_get_clock_output_frequencies(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_output_frequencies'{$endif};
function tiepie_hw_oscilloscope_get_clock_output_frequencies_ex(handle : Ttiepie_hw_handle; clock_output : LongWord; list : PDouble; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_output_frequencies_ex'{$endif};
function tiepie_hw_oscilloscope_get_clock_output_frequency(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_clock_output_frequency'{$endif};
function tiepie_hw_oscilloscope_set_clock_output_frequency(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_clock_output_frequency'{$endif};
function tiepie_hw_oscilloscope_get_sample_rate_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_sample_rate_max'{$endif};
function tiepie_hw_oscilloscope_get_sample_rate(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_sample_rate'{$endif};
function tiepie_hw_oscilloscope_set_sample_rate(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_sample_rate'{$endif};
function tiepie_hw_oscilloscope_verify_sample_rate(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_verify_sample_rate'{$endif};
function tiepie_hw_oscilloscope_verify_sample_rate_ex(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_verify_sample_rate_ex'{$endif};
procedure tiepie_hw_oscilloscope_verify_sample_rates_ex(handle : Ttiepie_hw_handle; values : PDouble; count, measure_mode, auto_resolution_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_verify_sample_rates_ex'{$endif};
function tiepie_hw_oscilloscope_get_record_length_max(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_record_length_max'{$endif};
function tiepie_hw_oscilloscope_get_record_length_max_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord; resolution : Byte) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_record_length_max_ex'{$endif};
function tiepie_hw_oscilloscope_get_record_length(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_record_length'{$endif};
function tiepie_hw_oscilloscope_set_record_length(handle : Ttiepie_hw_handle; record_length : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_record_length'{$endif};
function tiepie_hw_oscilloscope_verify_record_length(handle : Ttiepie_hw_handle; record_length : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_verify_record_length'{$endif};
function tiepie_hw_oscilloscope_verify_record_length_ex(handle : Ttiepie_hw_handle; record_length : UInt64; measure_mode : LongWord; resolution : Byte; channel_enabled : PByteBool; channel_count : Word) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_verify_record_length_ex'{$endif};
function tiepie_hw_oscilloscope_get_pre_sample_ratio(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_pre_sample_ratio'{$endif};
function tiepie_hw_oscilloscope_set_pre_sample_ratio(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_pre_sample_ratio'{$endif};
function tiepie_hw_oscilloscope_get_segment_count_max(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_segment_count_max'{$endif};
function tiepie_hw_oscilloscope_get_segment_count_max_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_segment_count_max_ex'{$endif};
function tiepie_hw_oscilloscope_get_segment_count(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_segment_count'{$endif};
function tiepie_hw_oscilloscope_set_segment_count(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_segment_count'{$endif};
function tiepie_hw_oscilloscope_verify_segment_count(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_verify_segment_count'{$endif};
function tiepie_hw_oscilloscope_verify_segment_count_ex(handle : Ttiepie_hw_handle; value, measure_mode : LongWord; record_length : UInt64; channel_enabled : PByteBool; channel_count : Word) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_verify_segment_count_ex'{$endif};
function tiepie_hw_oscilloscope_has_trigger(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_has_trigger'{$endif};
function tiepie_hw_oscilloscope_has_trigger_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_has_trigger_ex'{$endif};
function tiepie_hw_oscilloscope_trigger_get_timeout(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_get_timeout'{$endif};
function tiepie_hw_oscilloscope_trigger_set_timeout(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_set_timeout'{$endif};
function tiepie_hw_oscilloscope_trigger_verify_timeout(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_verify_timeout'{$endif};
function tiepie_hw_oscilloscope_trigger_verify_timeout_ex(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; sample_rate : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_verify_timeout_ex'{$endif};
function tiepie_hw_oscilloscope_trigger_has_delay(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_has_delay'{$endif};
function tiepie_hw_oscilloscope_trigger_has_delay_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_has_delay_ex'{$endif};
function tiepie_hw_oscilloscope_trigger_get_delay_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_get_delay_max'{$endif};
function tiepie_hw_oscilloscope_trigger_get_delay_max_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord; sample_rate : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_get_delay_max_ex'{$endif};
function tiepie_hw_oscilloscope_trigger_get_delay(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_get_delay'{$endif};
function tiepie_hw_oscilloscope_trigger_set_delay(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_set_delay'{$endif};
function tiepie_hw_oscilloscope_trigger_verify_delay(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_verify_delay'{$endif};
function tiepie_hw_oscilloscope_trigger_verify_delay_ex(handle : Ttiepie_hw_handle; value : Double; measure_mode : LongWord; sample_rate : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_trigger_verify_delay_ex'{$endif};
function tiepie_hw_oscilloscope_has_presamples_valid(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_has_presamples_valid'{$endif};
function tiepie_hw_oscilloscope_has_presamples_valid_ex(handle : Ttiepie_hw_handle; measure_mode : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_has_presamples_valid_ex'{$endif};
function tiepie_hw_oscilloscope_get_presamples_valid(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_presamples_valid'{$endif};
function tiepie_hw_oscilloscope_set_presamples_valid(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_set_presamples_valid'{$endif};
function tiepie_hw_oscilloscope_has_sureconnect(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_has_sureconnect'{$endif};
function tiepie_hw_oscilloscope_channel_has_sureconnect(handle : Ttiepie_hw_handle; ch : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_channel_has_sureconnect'{$endif};
function tiepie_hw_oscilloscope_start_sureconnect(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_start_sureconnect'{$endif};
function tiepie_hw_oscilloscope_start_sureconnect_ex(handle : Ttiepie_hw_handle; channel_enabled : PByteBool; channel_count : Word) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_start_sureconnect_ex'{$endif};
function tiepie_hw_oscilloscope_is_sureconnect_completed(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_is_sureconnect_completed'{$endif};
function tiepie_hw_oscilloscope_get_sureconnect_data(handle : Ttiepie_hw_handle; buffer : Ptiepie_hw_tristate; channel_count : Word) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_oscilloscope_get_sureconnect_data'{$endif};
function tiepie_hw_generator_get_connector_type(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_connector_type'{$endif};
function tiepie_hw_generator_is_differential(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_is_differential'{$endif};
function tiepie_hw_generator_get_impedance(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_impedance'{$endif};
function tiepie_hw_generator_get_resolution(handle : Ttiepie_hw_handle) : Byte; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_resolution'{$endif};
function tiepie_hw_generator_get_output_value_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_output_value_min'{$endif};
function tiepie_hw_generator_get_output_value_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_output_value_max'{$endif};
procedure tiepie_hw_generator_get_output_value_min_max(handle : Ttiepie_hw_handle; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_output_value_min_max'{$endif};
function tiepie_hw_generator_is_controllable(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_is_controllable'{$endif};
function tiepie_hw_generator_is_running(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_is_running'{$endif};
function tiepie_hw_generator_get_status(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_status'{$endif};
function tiepie_hw_generator_get_output_enable(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_output_enable'{$endif};
function tiepie_hw_generator_set_output_enable(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_output_enable'{$endif};
function tiepie_hw_generator_has_output_invert(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_output_invert'{$endif};
function tiepie_hw_generator_get_output_invert(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_output_invert'{$endif};
function tiepie_hw_generator_set_output_invert(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_output_invert'{$endif};
function tiepie_hw_generator_start(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_start'{$endif};
function tiepie_hw_generator_stop(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_stop'{$endif};
function tiepie_hw_generator_get_signal_types(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_signal_types'{$endif};
function tiepie_hw_generator_get_signal_type(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_signal_type'{$endif};
function tiepie_hw_generator_set_signal_type(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_signal_type'{$endif};
function tiepie_hw_generator_has_amplitude(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_amplitude'{$endif};
function tiepie_hw_generator_has_amplitude_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_amplitude_ex'{$endif};
function tiepie_hw_generator_get_amplitude_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_amplitude_min'{$endif};
function tiepie_hw_generator_get_amplitude_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_amplitude_max'{$endif};
procedure tiepie_hw_generator_get_amplitude_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_amplitude_min_max_ex'{$endif};
function tiepie_hw_generator_get_amplitude(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_amplitude'{$endif};
function tiepie_hw_generator_set_amplitude(handle : Ttiepie_hw_handle; amplitude : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_amplitude'{$endif};
function tiepie_hw_generator_verify_amplitude(handle : Ttiepie_hw_handle; amplitude : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_amplitude'{$endif};
function tiepie_hw_generator_verify_amplitude_ex(handle : Ttiepie_hw_handle; amplitude : Double; signal_type, amplitude_range_index : LongWord; offset : Double; output_invert : ByteBool) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_amplitude_ex'{$endif};
function tiepie_hw_generator_get_amplitude_ranges(handle : Ttiepie_hw_handle; list : PDouble; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_amplitude_ranges'{$endif};
function tiepie_hw_generator_get_amplitude_range(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_amplitude_range'{$endif};
function tiepie_hw_generator_set_amplitude_range(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_amplitude_range'{$endif};
function tiepie_hw_generator_get_amplitude_auto_ranging(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_amplitude_auto_ranging'{$endif};
function tiepie_hw_generator_set_amplitude_auto_ranging(handle : Ttiepie_hw_handle; value : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_amplitude_auto_ranging'{$endif};
function tiepie_hw_generator_has_offset(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_offset'{$endif};
function tiepie_hw_generator_has_offset_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_offset_ex'{$endif};
function tiepie_hw_generator_get_offset_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_offset_min'{$endif};
function tiepie_hw_generator_get_offset_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_offset_max'{$endif};
procedure tiepie_hw_generator_get_offset_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_offset_min_max_ex'{$endif};
function tiepie_hw_generator_get_offset(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_offset'{$endif};
function tiepie_hw_generator_set_offset(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_offset'{$endif};
function tiepie_hw_generator_verify_offset(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_offset'{$endif};
function tiepie_hw_generator_verify_offset_ex(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord; amplitude : Double; output_invert : ByteBool) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_offset_ex'{$endif};
function tiepie_hw_generator_get_frequency_modes(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_frequency_modes'{$endif};
function tiepie_hw_generator_get_frequency_modes_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_frequency_modes_ex'{$endif};
function tiepie_hw_generator_get_frequency_mode(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_frequency_mode'{$endif};
function tiepie_hw_generator_set_frequency_mode(handle : Ttiepie_hw_handle; value : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_frequency_mode'{$endif};
function tiepie_hw_generator_has_frequency(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_frequency'{$endif};
function tiepie_hw_generator_has_frequency_ex(handle : Ttiepie_hw_handle; frequency_mode, signal_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_frequency_ex'{$endif};
function tiepie_hw_generator_get_frequency_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_frequency_min'{$endif};
function tiepie_hw_generator_get_frequency_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_frequency_max'{$endif};
procedure tiepie_hw_generator_get_frequency_min_max(handle : Ttiepie_hw_handle; frequency_mode : LongWord; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_frequency_min_max'{$endif};
procedure tiepie_hw_generator_get_frequency_min_max_ex(handle : Ttiepie_hw_handle; frequency_mode, signal_type : LongWord; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_frequency_min_max_ex'{$endif};
function tiepie_hw_generator_get_frequency(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_frequency'{$endif};
function tiepie_hw_generator_set_frequency(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_frequency'{$endif};
function tiepie_hw_generator_verify_frequency(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_frequency'{$endif};
function tiepie_hw_generator_verify_frequency_ex(handle : Ttiepie_hw_handle; value : Double; frequency_mode, signal_type : LongWord; data_length : UInt64; width : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_frequency_ex'{$endif};
function tiepie_hw_generator_has_phase(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_phase'{$endif};
function tiepie_hw_generator_has_phase_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_phase_ex'{$endif};
function tiepie_hw_generator_get_phase_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_phase_min'{$endif};
function tiepie_hw_generator_get_phase_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_phase_max'{$endif};
procedure tiepie_hw_generator_get_phase_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_phase_min_max_ex'{$endif};
function tiepie_hw_generator_get_phase(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_phase'{$endif};
function tiepie_hw_generator_set_phase(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_phase'{$endif};
function tiepie_hw_generator_verify_phase(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_phase'{$endif};
function tiepie_hw_generator_verify_phase_ex(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_phase_ex'{$endif};
function tiepie_hw_generator_has_symmetry(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_symmetry'{$endif};
function tiepie_hw_generator_has_symmetry_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_symmetry_ex'{$endif};
function tiepie_hw_generator_get_symmetry_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_symmetry_min'{$endif};
function tiepie_hw_generator_get_symmetry_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_symmetry_max'{$endif};
procedure tiepie_hw_generator_get_symmetry_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_symmetry_min_max_ex'{$endif};
function tiepie_hw_generator_get_symmetry(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_symmetry'{$endif};
function tiepie_hw_generator_set_symmetry(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_symmetry'{$endif};
function tiepie_hw_generator_verify_symmetry(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_symmetry'{$endif};
function tiepie_hw_generator_verify_symmetry_ex(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_symmetry_ex'{$endif};
function tiepie_hw_generator_has_width(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_width'{$endif};
function tiepie_hw_generator_has_width_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_width_ex'{$endif};
function tiepie_hw_generator_get_width_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_width_min'{$endif};
function tiepie_hw_generator_get_width_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_width_max'{$endif};
procedure tiepie_hw_generator_get_width_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency : Double; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_width_min_max_ex'{$endif};
function tiepie_hw_generator_get_width(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_width'{$endif};
function tiepie_hw_generator_set_width(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_width'{$endif};
function tiepie_hw_generator_verify_width(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_width'{$endif};
function tiepie_hw_generator_verify_width_ex(handle : Ttiepie_hw_handle; value : Double; signal_type : LongWord; signal_frequency : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_width_ex'{$endif};
function tiepie_hw_generator_has_edge_time(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_edge_time'{$endif};
function tiepie_hw_generator_has_edge_time_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_edge_time_ex'{$endif};
function tiepie_hw_generator_get_leading_edge_time_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_leading_edge_time_min'{$endif};
function tiepie_hw_generator_get_leading_edge_time_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_leading_edge_time_max'{$endif};
procedure tiepie_hw_generator_get_leading_edge_time_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency, symmetry, width, trailing_edge_time : Double; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_leading_edge_time_min_max_ex'{$endif};
function tiepie_hw_generator_get_leading_edge_time(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_leading_edge_time'{$endif};
function tiepie_hw_generator_set_leading_edge_time(handle : Ttiepie_hw_handle; leading_edge_time : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_leading_edge_time'{$endif};
function tiepie_hw_generator_verify_leading_edge_time(handle : Ttiepie_hw_handle; leading_edge_time : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_leading_edge_time'{$endif};
function tiepie_hw_generator_verify_leading_edge_time_ex(handle : Ttiepie_hw_handle; leading_edge_time : Double; signal_type : LongWord; signal_frequency, symmetry, width, trailing_edge_time : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_leading_edge_time_ex'{$endif};
function tiepie_hw_generator_get_trailing_edge_time_min(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_trailing_edge_time_min'{$endif};
function tiepie_hw_generator_get_trailing_edge_time_max(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_trailing_edge_time_max'{$endif};
procedure tiepie_hw_generator_get_trailing_edge_time_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; signal_frequency, symmetry, width, leading_edge_time : Double; min, max : PDouble); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_trailing_edge_time_min_max_ex'{$endif};
function tiepie_hw_generator_get_trailing_edge_time(handle : Ttiepie_hw_handle) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_trailing_edge_time'{$endif};
function tiepie_hw_generator_set_trailing_edge_time(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_trailing_edge_time'{$endif};
function tiepie_hw_generator_verify_trailing_edge_time(handle : Ttiepie_hw_handle; value : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_trailing_edge_time'{$endif};
function tiepie_hw_generator_verify_trailing_edge_time_ex(handle : Ttiepie_hw_handle; trailing_edge_time : Double; signal_type : LongWord; signal_frequency, symmetry, width, leading_edge_time : Double) : Double; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_trailing_edge_time_ex'{$endif};
function tiepie_hw_generator_has_data(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_data'{$endif};
function tiepie_hw_generator_has_data_ex(handle : Ttiepie_hw_handle; signal_type : LongWord) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_has_data_ex'{$endif};
function tiepie_hw_generator_get_data_length_min(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_length_min'{$endif};
function tiepie_hw_generator_get_data_length_max(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_length_max'{$endif};
procedure tiepie_hw_generator_get_data_length_min_max_ex(handle : Ttiepie_hw_handle; signal_type : LongWord; min, max : PUInt64); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_length_min_max_ex'{$endif};
function tiepie_hw_generator_get_data_length(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_length'{$endif};
function tiepie_hw_generator_verify_data_length(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_data_length'{$endif};
function tiepie_hw_generator_verify_data_length_ex(handle : Ttiepie_hw_handle; value : UInt64; signal_type : LongWord) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_data_length_ex'{$endif};
procedure tiepie_hw_generator_set_data(handle : Ttiepie_hw_handle; buffer : PSingle; sample_count : UInt64); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_data'{$endif};
procedure tiepie_hw_generator_set_data_ex(handle : Ttiepie_hw_handle; buffer : PSingle; sample_count : UInt64; signal_type : LongWord); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_data_ex'{$endif};
function tiepie_hw_generator_get_data_raw_type(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_raw_type'{$endif};
procedure tiepie_hw_generator_get_data_raw_value_range(handle : Ttiepie_hw_handle; min, zero, max : PInt64); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_raw_value_range'{$endif};
function tiepie_hw_generator_get_data_raw_value_min(handle : Ttiepie_hw_handle) : Int64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_raw_value_min'{$endif};
function tiepie_hw_generator_get_data_raw_value_zero(handle : Ttiepie_hw_handle) : Int64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_raw_value_zero'{$endif};
function tiepie_hw_generator_get_data_raw_value_max(handle : Ttiepie_hw_handle) : Int64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_data_raw_value_max'{$endif};
procedure tiepie_hw_generator_set_data_raw(handle : Ttiepie_hw_handle; buffer : Pointer; sample_count : UInt64); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_data_raw'{$endif};
procedure tiepie_hw_generator_set_data_raw_ex(handle : Ttiepie_hw_handle; buffer : Pointer; sample_count : UInt64; signal_type : LongWord); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_data_raw_ex'{$endif};
function tiepie_hw_generator_get_modes(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_modes'{$endif};
function tiepie_hw_generator_get_modes_ex(handle : Ttiepie_hw_handle; signal_type, frequency_mode : LongWord) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_modes_ex'{$endif};
function tiepie_hw_generator_get_modes_native(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_modes_native'{$endif};
function tiepie_hw_generator_get_mode(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_mode'{$endif};
function tiepie_hw_generator_set_mode(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_mode'{$endif};
function tiepie_hw_generator_is_burst_active(handle : Ttiepie_hw_handle) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_is_burst_active'{$endif};
function tiepie_hw_generator_get_burst_count_min(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_count_min'{$endif};
function tiepie_hw_generator_get_burst_count_max(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_count_max'{$endif};
procedure tiepie_hw_generator_get_burst_count_min_max_ex(handle : Ttiepie_hw_handle; generator_mode : UInt64; min, max : PUInt64); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_count_min_max_ex'{$endif};
function tiepie_hw_generator_get_burst_count(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_count'{$endif};
function tiepie_hw_generator_set_burst_count(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_burst_count'{$endif};
function tiepie_hw_generator_get_burst_sample_count_min(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_sample_count_min'{$endif};
function tiepie_hw_generator_get_burst_sample_count_max(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_sample_count_max'{$endif};
procedure tiepie_hw_generator_get_burst_sample_count_min_max_ex(handle : Ttiepie_hw_handle; generator_mode : UInt64; min, max : PUInt64); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_sample_count_min_max_ex'{$endif};
function tiepie_hw_generator_get_burst_sample_count(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_sample_count'{$endif};
function tiepie_hw_generator_set_burst_sample_count(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_burst_sample_count'{$endif};
function tiepie_hw_generator_get_burst_segment_count_min(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_segment_count_min'{$endif};
function tiepie_hw_generator_get_burst_segment_count_max(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_segment_count_max'{$endif};
procedure tiepie_hw_generator_get_burst_segment_count_min_max_ex(handle : Ttiepie_hw_handle; generator_mode : UInt64; signal_type, frequency_mode : LongWord; frequency : Double; data_length : UInt64; min, max : PUInt64); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_segment_count_min_max_ex'{$endif};
function tiepie_hw_generator_get_burst_segment_count(handle : Ttiepie_hw_handle) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_get_burst_segment_count'{$endif};
function tiepie_hw_generator_set_burst_segment_count(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_set_burst_segment_count'{$endif};
function tiepie_hw_generator_verify_burst_segment_count(handle : Ttiepie_hw_handle; value : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_burst_segment_count'{$endif};
function tiepie_hw_generator_verify_burst_segment_count_ex(handle : Ttiepie_hw_handle; value, generator_mode : UInt64; signal_type, frequency_mode : LongWord; frequency : Double; data_length : UInt64) : UInt64; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_generator_verify_burst_segment_count_ex'{$endif};
function tiepie_hw_server_connect(handle : Ttiepie_hw_handle; asynchronous : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_connect'{$endif};
function tiepie_hw_server_disconnect(handle : Ttiepie_hw_handle; force : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_disconnect'{$endif};
function tiepie_hw_server_remove(handle : Ttiepie_hw_handle; force : ByteBool) : ByteBool; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_remove'{$endif};
function tiepie_hw_server_get_status(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_status'{$endif};
function tiepie_hw_server_get_last_error(handle : Ttiepie_hw_handle) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_last_error'{$endif};
function tiepie_hw_server_get_url(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_url'{$endif};
function tiepie_hw_server_get_id(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_id'{$endif};
function tiepie_hw_server_get_ip_address(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_ip_address'{$endif};
function tiepie_hw_server_get_ip_port(handle : Ttiepie_hw_handle) : Word; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_ip_port'{$endif};
function tiepie_hw_server_get_name(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_name'{$endif};
function tiepie_hw_server_get_description(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_description'{$endif};
function tiepie_hw_server_get_version(handle : Ttiepie_hw_handle; buffer : PAnsiChar; length : LongWord) : LongWord; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_server_get_version'{$endif};
function tiepie_hw_pointerarray_new(length : LongWord) : Ttiepie_hw_pointerarray; cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_pointerarray_new'{$endif};
procedure tiepie_hw_pointerarray_set(ptr : Ttiepie_hw_pointerarray; index : LongWord; value : Pointer); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_pointerarray_set'{$endif};
procedure tiepie_hw_pointerarray_delete(ptr : Ttiepie_hw_pointerarray); cdecl; external TIEPIE_HW_FILENAME_DEFAULT {$ifdef MACOS}name '_tiepie_hw_pointerarray_delete'{$endif};

{$endif}

//==============================================================================
// Extra routines

class operator Ttiepie_hw_date.Explicit(value: Ttiepie_hw_date): TDateTime;
begin
  if (value.year <> 0) and (value.month <> 0) and (value.day <> 0) then
    Result := EncodeDate(value.year, value.month, value.day)
  else
    Result := 0;
end;

class operator Ttiepie_hw_version.Implicit(value: Ttiepie_hw_version): Boolean;
begin
  Result := (value.major <> 0) or (value.minor <> 0) or (value.patch <> 0) or (value.build <> 0) or (value.extra <> '');
end;

function tiepie_hw_get_config_dynarray : TByteDynArray;
begin
  SetLength(Result, tiepie_hw_get_config(nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_get_config(@Result[0], Length(Result));
end;

function tiepie_hw_get_last_status_string() : string;
begin
  Result := UTF8ToString(tiepie_hw_get_last_status_str());
end;

function tiepie_hw_devicelistitem_opened_by_string(handle : Ttiepie_hw_handle; device_type : LongWord) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_devicelistitem_opened_by(handle , device_type, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_devicelistitem_opened_by(handle , device_type, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_devicelistitem_get_name_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_devicelistitem_get_name(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_devicelistitem_get_name(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_devicelistitem_get_name_short_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_devicelistitem_get_name_short(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_devicelistitem_get_name_short(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_devicelistitem_get_name_shortest_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_devicelistitem_get_name_shortest(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_devicelistitem_get_name_shortest(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_devicelistitem_get_ip_address_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_devicelistitem_get_ip_address(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_devicelistitem_get_ip_address(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_devicelistitem_get_contained_serial_numbers_dynarray(handle : Ttiepie_hw_handle) : TLongWordDynArray;
begin
  SetLength(Result, tiepie_hw_devicelistitem_get_contained_serial_numbers(handle , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_devicelistitem_get_contained_serial_numbers(handle , @Result[0], Length(Result));
end;

function tiepie_hw_devicelistitemcombined_get_name_string(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_devicelistitemcombined_get_name(handle , contained_device_serial_number, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_devicelistitemcombined_get_name(handle , contained_device_serial_number, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_devicelistitemcombined_get_name_short_string(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_devicelistitemcombined_get_name_short(handle , contained_device_serial_number, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_devicelistitemcombined_get_name_short(handle , contained_device_serial_number, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_devicelistitemcombined_get_name_shortest_string(handle : Ttiepie_hw_handle; contained_device_serial_number : LongWord) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_devicelistitemcombined_get_name_shortest(handle , contained_device_serial_number, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_devicelistitemcombined_get_name_shortest(handle , contained_device_serial_number, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_device_get_ip_address_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_device_get_ip_address(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_device_get_ip_address(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_device_get_name_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_device_get_name(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_device_get_name(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_device_get_name_short_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_device_get_name_short(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_device_get_name_short(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_device_get_name_shortest_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_device_get_name_shortest(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_device_get_name_shortest(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_device_trigger_input_get_name_string(handle : Ttiepie_hw_handle; input : Word) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_device_trigger_input_get_name(handle , input, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_device_trigger_input_get_name(handle , input, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_device_trigger_output_get_name_string(handle : Ttiepie_hw_handle; output : Word) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_device_trigger_output_get_name(handle , output, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_device_trigger_output_get_name(handle , output, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_oscilloscope_channel_get_bandwidths_dynarray(handle : Ttiepie_hw_handle; ch : Word) : TDoubleDynArray;
begin
  SetLength(Result, tiepie_hw_oscilloscope_channel_get_bandwidths(handle , ch , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_oscilloscope_channel_get_bandwidths(handle , ch , @Result[0], Length(Result));
end;

function tiepie_hw_oscilloscope_channel_get_ranges_dynarray(handle : Ttiepie_hw_handle; ch : Word) : TDoubleDynArray;
begin
  SetLength(Result, tiepie_hw_oscilloscope_channel_get_ranges(handle , ch , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_oscilloscope_channel_get_ranges(handle , ch , @Result[0], Length(Result));
end;

function tiepie_hw_oscilloscope_channel_get_ranges_ex_dynarray(handle : Ttiepie_hw_handle; ch : Word; coupling : UInt64) : TDoubleDynArray;
begin
  SetLength(Result, tiepie_hw_oscilloscope_channel_get_ranges_ex(handle , ch , coupling , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_oscilloscope_channel_get_ranges_ex(handle , ch , coupling , @Result[0], Length(Result));
end;

function tiepie_hw_oscilloscope_get_resolutions_dynarray(handle : Ttiepie_hw_handle) : TByteDynArray;
begin
  SetLength(Result, tiepie_hw_oscilloscope_get_resolutions(handle , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_oscilloscope_get_resolutions(handle , @Result[0], Length(Result));
end;

function tiepie_hw_oscilloscope_get_clock_source_frequencies_dynarray(handle : Ttiepie_hw_handle) : TDoubleDynArray;
begin
  SetLength(Result, tiepie_hw_oscilloscope_get_clock_source_frequencies(handle , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_oscilloscope_get_clock_source_frequencies(handle , @Result[0], Length(Result));
end;

function tiepie_hw_oscilloscope_get_clock_source_frequencies_ex_dynarray(handle : Ttiepie_hw_handle; value : LongWord) : TDoubleDynArray;
begin
  SetLength(Result, tiepie_hw_oscilloscope_get_clock_source_frequencies_ex(handle , value , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_oscilloscope_get_clock_source_frequencies_ex(handle , value , @Result[0], Length(Result));
end;

function tiepie_hw_oscilloscope_get_clock_output_frequencies_dynarray(handle : Ttiepie_hw_handle) : TDoubleDynArray;
begin
  SetLength(Result, tiepie_hw_oscilloscope_get_clock_output_frequencies(handle , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_oscilloscope_get_clock_output_frequencies(handle , @Result[0], Length(Result));
end;

function tiepie_hw_oscilloscope_get_clock_output_frequencies_ex_dynarray(handle : Ttiepie_hw_handle; clock_output : LongWord) : TDoubleDynArray;
begin
  SetLength(Result, tiepie_hw_oscilloscope_get_clock_output_frequencies_ex(handle , clock_output , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_oscilloscope_get_clock_output_frequencies_ex(handle , clock_output , @Result[0], Length(Result));
end;

function tiepie_hw_generator_get_amplitude_ranges_dynarray(handle : Ttiepie_hw_handle) : TDoubleDynArray;
begin
  SetLength(Result, tiepie_hw_generator_get_amplitude_ranges(handle , nil, 0));
  if Length(Result) > 0 then
    tiepie_hw_generator_get_amplitude_ranges(handle , @Result[0], Length(Result));
end;

function tiepie_hw_server_get_url_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_server_get_url(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_server_get_url(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_server_get_id_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_server_get_id(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_server_get_id(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_server_get_ip_address_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_server_get_ip_address(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_server_get_ip_address(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_server_get_name_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_server_get_name(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_server_get_name(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_server_get_description_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_server_get_description(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_server_get_description(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

function tiepie_hw_server_get_version_string(handle : Ttiepie_hw_handle) : string;
var
  tmp : AnsiString;
  length : LongWord;
begin
  length := tiepie_hw_server_get_version(handle, nil, 0) + 1;
  if length > 0 then begin
    SetLength(tmp, length);
    SetLength(tmp, tiepie_hw_server_get_version(handle, @tmp[1], length));
    Result := UTF8ToString(tmp);
  end;
end;

// END Extra routines
//==============================================================================

{$ifndef LIBTIEPIE_HW_DYNAMIC}
initialization
  tiepie_hw_init();
finalization
  tiepie_hw_fini();
{$endif}
end.

