/* GeneratorArbitrary.cs
 *
 * This example generates an arbitrary waveform.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open a generator with arbitrary support:
Generator? gen = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using var item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Generator))
    {
        gen = item.OpenGenerator();

        // Check for arbitrary support:
        if (gen.SignalTypes.HasFlag(SignalType.Arbitrary))
            break;
        else
        {
            gen.Dispose();
            gen = null;
        }
    }
}

if (gen is null)
{
    Console.WriteLine("No generator available with arbitrary support!");
    return 1;
}

try
{
    // Set signal type:
    gen.SignalType = SignalType.Arbitrary;

    // Select frequency mode:
    gen.FrequencyMode = FrequencyMode.SampleRate;

    // Set frequency:
    gen.Frequency = 100e3; // 100 kHz

    // Set amplitude:
    gen.Amplitude = 2; // 2 V

    // Set offset:
    gen.Offset = 0; // 0 V

    // Enable output:
    gen.OutputEnable = true;

    // Create signal array:
    var data = new float[8192];
    for (var i = 0; i < data.Length; i++)
        data[i] = (float)(Math.Sin(i / 100.0) * (1 - (i / 8192.0)));

    // Load the signal array into the generator:
    gen.SetData(data);

    // Print generator info:
    PrintInfo.PrintDeviceInfo(gen);

    // Start signal generation:
    gen.Start();

    // Wait for keystroke:
    Console.WriteLine("Press Enter to stop signal generation...");
    Console.ReadLine();

    // Stop generator:
    gen.Stop();

    // Disable output:
    gen.OutputEnable = false;
}
catch (System.Exception e)
{
    Console.WriteLine($"Exception: {e.Message}");
    return 1;
}
finally
{
    // Close generator:
    gen.Dispose();
}

return 0;
