/* GeneratorBurst.cs
 *
 * This example generates a 50 Hz sine waveform, 4 Vpp, 100 periods.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using System.Threading;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open a generator with burst support:
Generator? gen = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using var item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Generator))
    {
        gen = item.OpenGenerator();

        // Check for burst support:
        if (gen.ModesNative.HasFlag(GeneratorMode.BurstCount))
            break;
        else
        {
            gen.Dispose();
            gen = null;
        }
    }
}

if (gen is null)
{
    Console.WriteLine("No generator available with burst support!");
    return 1;
}

try
{
    // Set signal type:
    gen.SignalType = SignalType.Sine;

    // Set frequency:
    gen.Frequency = 50; // 50 Hz

    // Set amplitude:
    gen.Amplitude = 2; // 2 V

    // Set offset:
    gen.Offset = 0; // 0 V

    // Set burst mode:
    gen.Mode = GeneratorMode.BurstCount;

    // Set burst count:
    gen.BurstCount = 100; // 100 periods

    // Enable output:
    gen.OutputEnable = true;

    // Print Generator info:
    PrintInfo.PrintDeviceInfo(gen);

    // Start signal generation
    gen.Start();

    // Wait for burst to complete:
    while (gen.IsBurstActive)
        Thread.Sleep(10); // 10 ms delay, to save CPU time.

    // Disable output:
    gen.OutputEnable = false;
}
catch (System.Exception e)
{
    Console.WriteLine($"Exception: {e.Message}");
    return 1;
}
finally
{
    // Close generator:
    gen.Dispose();
}

return 0;
