/* GeneratorGatedBurst.cs
 *
 * This example generates a 10 kHz square waveform, 10 Vpp when the external trigger input is active.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open a generator with gated burst support:
Generator? gen = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using var item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Generator))
    {
        gen = item.OpenGenerator();

        // Check for gated burst support:
        if (gen.ModesNative.HasFlag(GeneratorMode.GatedPeriods) && gen.TriggerInputs.Count > 0)
            break;
        else
        {
            gen.Dispose();
            gen = null;
        }
    }
}

if (gen is null)
{
    Console.WriteLine("No generator available with gated burst support!");
    return 1;
}

try
{
    // Set signal type:
    gen.SignalType = SignalType.Square;

    // Set frequency:
    gen.Frequency = 10e3; // 10 kHz

    // Set amplitude:
    gen.Amplitude = 5; // 5 V

    // Set offset:
    gen.Offset = 0; // 0 V

    // Set mode:
    gen.Mode = GeneratorMode.GatedPeriods;

    // Locate trigger input:
    TriggerInput triggerInput = gen.TriggerInputs.GetByID(TriggerInputID.Ext1) ?? throw new System.Exception("Unknown trigger input!");

    // Enable trigger input:
    triggerInput.Enabled = true;

    // Enable output:
    gen.OutputEnable = true;

    // Print Generator info:
    PrintInfo.PrintDeviceInfo(gen);

    // Start signal generation
    gen.Start();

    // Wait for keystroke:
    Console.WriteLine("Press Enter to stop signal generation...");
    Console.ReadLine();

    // Stop generator:
    gen.Stop();

    // Disable output:
    gen.OutputEnable = false;

}
catch (System.Exception e)
{
    Console.WriteLine($"Exception: {e.Message}");
    return 1;
}
finally
{
    // Close generator:
    gen.Dispose();
}

return 0;
