/* GeneratorTriggeredBurst.cs - for LibTiePie.NET 0.9 or newer
 *
 * This example generates a 100 kHz square waveform, 25% duty cycle, 0..5 V, 20 periods, this waveform is triggered by the external trigger.
 * Connect the external trigger to GND to trigger the burst.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open a generator with triggered burst support:
Generator? gen = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using var item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Generator))
    {
        gen = item.OpenGenerator();

        // Check for triggered burst support:
        if (gen.ModesNative.HasFlag(GeneratorMode.BurstCount) && gen.TriggerInputs.Count > 0)
            break;
        else
        {
            gen.Dispose();
            gen = null;
        }
    }
}

if (gen is null)
{
    Console.WriteLine("No generator available with triggered burst support!");
    return 1;
}

try
{
    // Set signal type:
    gen.SignalType = SignalType.Square;

    // Set frequency:
    gen.Frequency = 100e3; // 100 kHz

    // Set amplitude:
    gen.Amplitude = 2.5; // 2.5 V

    // Set offset:
    gen.Offset = 2.5; // 2.5 V

    // Set symmetry (duty cycle):
    gen.Symmetry = 0.25; // 25 %

    // Set burst mode:
    gen.Mode = GeneratorMode.BurstCount;

    // Set burst count:
    gen.BurstCount = 20; // 20 periods

    // Locate trigger input:
    TriggerInput? triggerInput = gen.TriggerInputs.GetByID(TriggerInputID.Ext1);

    if (triggerInput is null)
        triggerInput = gen.TriggerInputs.GetByID(TriggerInputID.Ext2);

    if (triggerInput is null)
        throw new System.Exception("Unknown trigger input!");

    // Enable trigger input:
    triggerInput.Enabled = true;

    // Set trigger input kind:
    triggerInput.Kind = TriggerKind.FallingEdge;

    // Enable output:
    gen.OutputEnable = true;

    // Print generator info:
    PrintInfo.PrintDeviceInfo(gen);

    // Start signal generation
    gen.Start();

    // Wait for keystroke:
    Console.WriteLine("Press Enter to stop signal generation...");
    Console.ReadLine();

    // Stop generator:
    gen.Stop();

    // Disable output:
    gen.OutputEnable = false;
}
catch (System.Exception e)
{
    Console.WriteLine($"Exception: {e.Message}");
    return 1;
}
finally
{
    // Close generator:
    gen.Dispose();
}

return 0;
