/* OscilloscopeBlock.cs
 *
 * This example performs a block mode measurement and writes the data to OscilloscopeBlock.csv.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using System.IO;
using System.Linq;
using System.Threading;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open an oscilloscope with block measurement support:
Oscilloscope? scp = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using var item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Oscilloscope))
    {
        scp = item.OpenOscilloscope();

        // Check for block measurement support:
        if (scp.MeasureModes.HasFlag(MeasureMode.Block))
            break;
        else
        {
            scp.Dispose();
            scp = null;
        }
    }
}

if (scp is null)
{
    Console.WriteLine("No oscilloscope available with block measurement support!");
    return 1;
}

try
{
    // Get the number of channels:
    var channelCount = scp.Channels.Count;

    // Set measure mode:
    scp.MeasureMode = MeasureMode.Block;

    // Set sample rate:
    scp.SampleRate = 1e6; // 1 MHz

    // Set record length:
    scp.RecordLength = 10000; // 10 kS
    var recordLength = scp.RecordLength; // Read actual record length.

    // Set pre sample ratio:
    scp.PreSampleRatio = 0; // 0 %

    // For all channels:
    foreach (var ch in scp.Channels)
    {
        if (!ch.IsAvailable)
            continue;

        // Enable channel to measure it:
        ch.Enabled = true;

        // Set range:
        ch.Range = 8; // 8 V

        // Set coupling:
        ch.Coupling = Coupling.DCV; // DC Volt
    }

    // Set trigger timeout:
    scp.Trigger.TimeOut = 100e-3; // 100 ms

    // Disable all channel trigger sources:
    foreach (var ch in scp.Channels)
    {
        if (ch.Trigger is not null)
            ch.Trigger.Enabled = false;
    }

    // Setup channel trigger:
    var channelTrigger = scp.Channels[0].Trigger; // Ch 1

    if (channelTrigger is not null)
    {
        // Enable trigger source:
        channelTrigger.Enabled = true;

        // Kind:
        channelTrigger.Kind = TriggerKind.RisingEdge;

        // Level:
        channelTrigger.Level[0] = 0.5; // 50 %

        // Hysteresis:
        channelTrigger.Hysteresis[0] = 0.05; // 5 %
    }

    // Print oscilloscope info:
    PrintInfo.PrintDeviceInfo(scp);

    // Start measurement:
    scp.Start();

    // Wait for measurement to complete:
    while (!scp.IsDataReady)
        Thread.Sleep(10); // 10 ms delay, to save CPU time.

    // Get data:
    var data = scp.GetData();

    // Write the data to csv:
    var filename = "OscilloscopeBlock.csv";
    using var file = new StreamWriter(filename);

    // Write csv header:
    file.WriteLine($"Sample,Ch{string.Join(",Ch", Enumerable.Range(1, channelCount))}");

    // Write the data to csv:
    for (var i = 0ul; i < recordLength; i++)
    {
        file.Write($"{i}");
        for (var ch = 0; ch < channelCount; ch++)
        {
            if (data[ch] is not null)
                file.Write($",{data[ch][i]}");
        }
        file.Write(Environment.NewLine);
    }

    Console.WriteLine($"Data written to: {filename}");
}
catch (System.Exception e)
{
    Console.WriteLine($"Exception: {e.Message}");
    return 1;
}
finally
{
    // Close oscilloscope:
    scp.Dispose();
}

return 0;
