/* OscilloscopeBlockSegmented.cs
 *
 * This example performs a block mode measurement of 5 segments and writes the data to OscilloscopeBlockSegmented.csv.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open an oscilloscope with block measurement support:
Oscilloscope? scp = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using var item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Oscilloscope))
    {
        scp = item.OpenOscilloscope();

        // Check for block measurement support:
        if (scp.MeasureModes.HasFlag(MeasureMode.Block) & (scp.SegmentCountMax > 1))
            break;
        else
        {
            scp.Dispose();
            scp = null;
        }
    }
}

if (scp is null)
{
    Console.WriteLine("No oscilloscope available with block measurement and segmented trigger support!");
    return 1;
}

try
{
    // Set measure mode:
    scp.MeasureMode = MeasureMode.Block;

    // Set sample rate:
    scp.SampleRate = 1e6; // 1 MHz

    // Set record length:
    scp.RecordLength = 1000; // 1 kS
    var recordLength = scp.RecordLength; // Read actual record length.

    // Set pre sample ratio:
    scp.PreSampleRatio = 0; // 0 %

    // Set number of segments:
    scp.SegmentCount = 5; // 5 segments

    // For all channels:
    foreach (var ch in scp.Channels)
    {
        if (!ch.IsAvailable)
            continue;

        // Disable channel:
        ch.Enabled = false;

        // Set range:
        ch.Range = 8; // 8 V

        // Set coupling:
        ch.Coupling = Coupling.DCV; // DC Volt
    }

    // Enable channel to measure it:
    scp.Channels[0].Enabled = true;

    // Set trigger timeout:
    scp.Trigger.TimeOut = 100e-3; // 100 ms

    // Disable all channel trigger sources:
    foreach (var ch in scp.Channels)
    {
        if(ch.Trigger is not null)
            ch.Trigger.Enabled = false;
    }

    // Setup channel trigger:
    var channelTrigger = scp.Channels[0].Trigger; // Ch 1

    if (channelTrigger is not null)
    {
        // Enable trigger source:
        channelTrigger.Enabled = true;

        // Kind:
        channelTrigger.Kind = TriggerKind.RisingEdge;

        // Level:
        channelTrigger.Level[0] = 0.5; // 50 %

        // Hysteresis:
        channelTrigger.Hysteresis[0] = 0.05; // 5 %
    }

    // Print oscilloscope info:
    PrintInfo.PrintDeviceInfo(scp);

    // Start measurement:
    scp.Start();

    // Wait for measurement to complete:
    while (!scp.IsDataReady)
        Thread.Sleep(10); // 10 ms delay, to save CPU time.

    // Get all data from the scope:
    var data = new List<float[][]>((int)scp.SegmentCount);
    while (scp.IsDataReady)
        data.Add(scp.GetData());

    // Write the data to csv:
    var filename = "OscilloscopeBlockSegmented.csv";
    using var file = new StreamWriter(filename);

    // Write csv header:
    file.WriteLine($"Sample,{string.Join(',', Enumerable.Range(1, data.Count))}");

    // Write the Ch1 data to csv:
    for (var i = 0ul; i < recordLength; i++)
    {
        file.Write(i.ToString());
        foreach (var segment in data)
            file.Write($",{segment[0][i]}");
        file.Write(Environment.NewLine);
    }

    Console.WriteLine($"Data written to: {filename}");
}
catch (System.Exception e)
{
    Console.WriteLine($"Exception: {e.Message}");
    return 1;
}
finally
{
    // Close oscilloscope:
    scp.Dispose();
}

return 0;
