/* OscilloscopeConnectionTest.cs
 *
 * This example performs a connection test.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using System.Threading;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open an oscilloscope with connection test support:
Oscilloscope? scp = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using var item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Oscilloscope))
    {
        scp = item.OpenOscilloscope();

        // Check for connection test support:
        if (scp.HasSureConnect)
            break;
        else
        {
            scp.Dispose();
            scp = null;
        }
    }
}

if (scp is not null)
{
    try
    {
        // Get the number of channels:
        var channelCount = scp.Channels.Count;

        // Enable all channels that support connection testing:
        foreach (var ch in scp.Channels)
            ch.Enabled = ch.HasSureConnect;

        // Start connection on current active channels:
        scp.StartSureConnect();

        // Wait for connectiontest to complete:
        while (!scp.IsSureConnectCompleted)
            Thread.Sleep(10); // 10 ms delay, to save CPU time.

        // Get data:
        var data = scp.GetSureConnectData();

        // Print results:
        Console.WriteLine("Connection test result:");
        for (var ch = 0; ch < channelCount; ch++)
        {
            Console.Write($"Ch{ch + 1} = ");

            switch (data[ch])
            {
                case TriState.Undefined:
                    Console.WriteLine("undefined");
                    break;

                case TriState.False:
                    Console.WriteLine("false");
                    break;

                case TriState.True:
                    Console.WriteLine("true");
                    break;

                default:
                    Console.WriteLine("unknown state");
                    break;
            }
        }
    }
    catch (System.Exception e)
    {
        Console.WriteLine($"Exception: {e.Message}");
        return 1;
    }
    finally
    {
        // Close oscilloscope:
        scp.Dispose();
    }
}
else
{
    Console.WriteLine("No oscilloscope available with connection test support!");
    return 1;
}

return 0;
