﻿/* OscilloscopeGeneratorTrigger.cs
 *
 * This example sets up the generator to generate a 1 kHz triangle waveform, 4 Vpp.
 * It also sets up the oscilloscope to perform a block mode measurement, triggered on "Generator new period".
 * A measurement is performed and the data is written to OscilloscopeGeneratorTrigger.csv.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using System.IO;
using System.Linq;
using System.Threading;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open an oscilloscope with block measurement support and a generator in the same device:
Oscilloscope? scp = null;
Generator? gen = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using var item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Oscilloscope) && item.CanOpen(DeviceType.Generator))
    {
        scp = item.OpenOscilloscope();

        if (scp.MeasureModes.HasFlag(MeasureMode.Block))
        {
            gen = item.OpenGenerator();
            break;
        }
        else
        {
            scp.Dispose();
            scp = null;
        }
    }
}

if (scp is null || gen is null)
{
    Console.WriteLine("No oscilloscope available with block measurement support or generator available in the same unit!");
    return 1;
}

try
{
    // Oscilloscope settings:

    // Get the number of channels:
    var channelCount = Convert.ToUInt16(scp.Channels.Count);

    // Set measure mode:
    scp.MeasureMode = MeasureMode.Block;

    // Set sample frequency:
    scp.SampleRate = 1e6; // 1 MHz

    // Set record length:
    scp.RecordLength = 10000; // 10 kS
    var recordLength = scp.RecordLength; // Read actual record length.

    // Set pre sample ratio:
    scp.PreSampleRatio = 0; // 0 %

    // For all channels:
    foreach (var ch in scp.Channels)
    {
        // Enable channel to measure it:
        ch.Enabled = true;

        // Set range:
        ch.Range = 8; // 8 V

        // Set coupling:
        ch.Coupling = Coupling.DCV; // DC Volt
    }

    // Set trigger timeout:
    scp.Trigger.TimeOut = 1; // 1 s

    // Disable all channel trigger sources:
    foreach (var ch in scp.Channels)
    {
        if (ch.Trigger is not null)    
            ch.Trigger.Enabled = false;
    }

    // Locate trigger input:
    TriggerInput triggerInput = scp.TriggerInputs.GetByID(TriggerInputID.GeneratorNewPeriod) ?? throw new System.Exception("Unknown trigger input!"); // or TriggerInputID.GeneratorStart or TriggerInputID.GeneratorStop

    // Enable trigger input:
    triggerInput.Enabled = true;

    // Generator settings:

    // Set signal type:
    gen.SignalType = SignalType.Triangle;

    // Set frequency:
    gen.Frequency = 1e3; // 1 kHz

    // Set amplitude:
    gen.Amplitude = 2; // 2 V

    // Set offset:
    gen.Offset = 0; // 0 V

    // Enable output:
    gen.OutputEnable = true;

    // Print oscilloscope info:
    PrintInfo.PrintDeviceInfo(scp);

    // Print generator info:
    PrintInfo.PrintDeviceInfo(gen);

    // Start measurement:
    scp.Start();

    // Start signal generation:
    gen.Start();

    // Wait for measurement to complete:
    while (!scp.IsDataReady)
        Thread.Sleep(10); // 10 ms delay, to save CPU time.

    // Stop generator:
    gen.Stop();

    // Disable output:
    gen.OutputEnable = false;

    // Get data:
    var data = scp.GetData();

    // Output CSV data:
    var filename = "OscilloscopeGeneratorTrigger.csv";
    using var file = new StreamWriter(filename);

    // Write csv header:
    file.WriteLine($"Sample,Ch{string.Join(",Ch", Enumerable.Range(1, channelCount))}");

    // Write the data to csv:
    for (var i = 0ul; i < recordLength; i++)
    {
        file.Write($"{i}");
        for (var ch = 0; ch < channelCount; ch++)
            file.Write($",{data[ch][i]}");
        file.Write(Environment.NewLine);
    }

    Console.WriteLine($"Data written to: {filename}");
}
catch (System.Exception e)
{
    Console.WriteLine($"Exception: {e.Message}");
    return 1;
}
finally
{
    // Close oscilloscope:
    scp.Dispose();

    // Close generator:
    gen.Dispose();
}

return 0;
