/* OscilloscopeStream.cs
 *
 * This example performs a stream mode measurement and writes the data to OscilloscopeStream.csv.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using System.IO;
using System.Linq;
using System.Threading;
using TiePie.Hardware;

// Print library information:
PrintInfo.PrintLibraryInfo();

// Enable network search:
Network.AutoDetectEnabled = true;

// Update device list:
DeviceList.Update();

// Try to open an oscilloscope with with stream measurement support:
Oscilloscope? scp = null;

for (var i = 0u; i < DeviceList.Count; i++)
{
    using DeviceListItem item = DeviceList.GetItemByIndex(i);
    if (item.CanOpen(DeviceType.Oscilloscope))
    {
        scp = item.OpenOscilloscope();

        // Check for stream measurement support:
        if (scp.MeasureModes.HasFlag(MeasureMode.Stream))
            break;
        else
        {
            scp.Dispose();
            scp = null;
        }
    }
}

if (scp is null)
{
    Console.WriteLine("No oscilloscope available with stream measurement support!");
    return 1;
}

try
{
    // Get the number of channels:
    var channelCount = scp.Channels.Count;

    // Set measure mode:
    scp.MeasureMode = MeasureMode.Stream;

    // Set sample frequency:
    scp.SampleRate = 1e3; // 1 kHz

    // Set record length:
    scp.RecordLength = 1000; // 1 kS
    var recordLength = scp.RecordLength; // Read actual record length.

    // For all channels:
    foreach (var ch in scp.Channels)
    {
        // Enable channel to measure it:
        ch.Enabled = true;

        // Set range:
        ch.Range = 8; // 8 V

        // Set coupling:
        ch.Coupling = Coupling.DCV; // DC Volt
    }

    // Print oscilloscope info:
    PrintInfo.PrintDeviceInfo(scp);

    // Open file with write permissions:
    var filename = "OscilloscopeStream.csv";
    using var file = new StreamWriter(filename);

    // Start measurement:
    scp.Start();

    // Write csv header:
    file.WriteLine($"Sample,Ch{string.Join(",Ch", Enumerable.Range(1, channelCount))}");

    var currentSample = 0ul;

    for (var chunk = 0; chunk <= 9; chunk++) // Measure 10 chunks.
    {
        // Print a message, to inform the user that we still do something:
        Console.WriteLine($"Data chunk {chunk}");

        // Wait for measurement to complete:
        while (!(scp.IsDataReady || scp.IsDataOverflow))
            Thread.Sleep(10); // 10 ms delay, to save CPU time.

        // Throw error on data overflow:
        if (scp.IsDataOverflow)
            throw new System.Exception("Data overflow!");

        // Get data:
        var data = scp.GetData();

        // Write the data to csv:
        for (var i = 0ul; i < recordLength; i++)
        {
            file.Write(currentSample + i);
            for (var ch = 0; ch < channelCount; ch++)
                file.Write($",{data[ch][i]}");
            file.Write(Environment.NewLine);
        }

        currentSample += recordLength;
    }

    Console.WriteLine($"Data written to: {filename}");

    // Stop measurement:
    scp.Stop();
}
catch (System.Exception e)
{
    Console.WriteLine($"Exception: {e.Message}");
    return 1;
}
finally
{
    // Close oscilloscope:
    scp.Dispose();
}

return 0;
