/**
 * PrintInfo.cs
 *
 * This file is part of the LibTiePie programming examples.
 *
 * Find more information on http://www.tiepie.com/LibTiePie .
 */

using System;
using TiePie.Hardware;

public static class PrintInfo
{
    public static void PrintLibraryInfo()
    {
        try
        {
            Console.WriteLine("Library:");
            Console.WriteLine($"  Version      :  {Library.Version}");
            Console.WriteLine($"  Configuration:  {Library.ConfigStr}");
        }
        catch (System.Exception e)
        {
            Console.WriteLine($"Exception: {e.Message}");
        }
    }

    public static void PrintDeviceInfo(Device dev)
    {
        try
        {
            Console.WriteLine("Device:");
            Console.WriteLine($"  Name                      : {dev.Name}");
            Console.WriteLine($"  Short name                : {dev.NameShort}");
            Console.WriteLine($"  Serial number             : {dev.SerialNumber}");

            try
            {
                Console.WriteLine($"  Calibration date          : {dev.CalibrationDate}");
            }
            catch (System.Exception)
            {
            }

            Console.WriteLine($"  Product id                : {dev.ProductID}");

            try
            {
                Console.WriteLine($"  IP address                : {dev.IpAddress}");
            }
            catch (System.Exception)
            {
            }

            try
            {
                Console.WriteLine($"  IP port                   : {dev.IpPort}");
            }
            catch (System.Exception)
            {
            }

            Console.WriteLine($"  Has battery               : {dev.HasBattery}");
            if (dev.HasBattery)
            {
                Console.WriteLine("  Battery:");
                try
                {
                    Console.WriteLine($"    Charge                  : {dev.BatteryCharge} %");
                }
                catch (System.Exception)
                {
                }

                try
                {
                    Console.WriteLine($"    Time to empty           : {dev.BatteryTimeToEmpty} minutes");
                }
                catch (System.Exception)
                {
                }

                try
                {
                    Console.WriteLine($"    Time to full            : {dev.BatteryTimeToFull} minutes");
                }
                catch (System.Exception)
                {
                }

                try
                {
                    Console.WriteLine($"    Charger connected       : {dev.IsBatteryChargerConnected}");
                }
                catch (System.Exception)
                {
                }

                try
                {
                    Console.WriteLine($"    Charging                : {dev.IsBatteryCharging}");
                }
                catch (System.Exception)
                {
                }

                try
                {
                    Console.WriteLine($"    Broken                  : {dev.IsBatteryBroken}");
                }
                catch (System.Exception)
                {
                }
            }

            if (dev is Oscilloscope scp)
                PrintOscilloscopeInfo(scp);

            if (dev is Generator gen)
                PrintGeneratorInfo(gen);
        }
        catch (System.Exception e)
        {
            Console.WriteLine($"Exception: {e.Message}");
        }
    }

    public static void PrintOscilloscopeInfo(Oscilloscope scp)
    {
        try
        {
            Console.WriteLine("Oscilloscope:");
            Console.WriteLine($"  Channel count             : {scp.Channels.Count}");
            Console.WriteLine($"  SureConnect               : {scp.HasSureConnect}");
            Console.WriteLine($"  Measure modes             : {scp.MeasureModes}");
            Console.WriteLine($"  Measure mode              : {scp.MeasureMode}");
            Console.WriteLine($"  Auto resolution modes     : {scp.AutoResolutionModes}");
            Console.WriteLine($"  Auto resolution mode      : {scp.AutoResolutionMode}");
            Console.WriteLine($"  Resolutions               : {string.Join(", ", scp.Resolutions)}");
            Console.WriteLine($"  Resolution                : {scp.Resolution}");
            Console.WriteLine($"  Resolution enhanced       : {scp.IsResolutionEnhanced}");
            Console.WriteLine($"  Clock outputs             : {scp.ClockOutputs}");
            Console.WriteLine($"  Clock output              : {scp.ClockOutput}");
            try
            {
                Console.WriteLine($"  Clock output frequencies  : {string.Join(", ", scp.ClockOutputFrequencies)}");
                Console.WriteLine($"  Clock output frequency    : {scp.ClockOutputFrequency}");
            }
            catch
            {
            }
            Console.WriteLine($"  Clock sources             : {scp.ClockSources}");
            Console.WriteLine($"  Clock source              : {scp.ClockSource}");
            try
            {
                Console.WriteLine($"  Clock output frequencies  : {string.Join(", ", scp.ClockOutputFrequencies)}");
                Console.WriteLine($"  Clock output frequency    : {scp.ClockOutputFrequency}");
            }
            catch
            {
            }

            Console.WriteLine($"  Record length max         : {scp.RecordLengthMax}");
            Console.WriteLine($"  Record length             : {scp.RecordLength}");
            Console.WriteLine($"  Sample rate max           : {scp.SampleRateMax}");
            Console.WriteLine($"  Sample rate               : {scp.SampleRate}");

            if (scp.MeasureMode == MeasureMode.Block)
            {
                Console.WriteLine($"  Segment count max         : {scp.SegmentCountMax}");
                if (scp.SegmentCountMax > 1)
                    Console.WriteLine($"  Segment count             : {scp.SegmentCount}");
            }

            if (scp.HasTrigger)
            {
                Console.WriteLine($"  Pre sample ratio          : {scp.PreSampleRatio}");
                Console.WriteLine($"  Trigger time out          : {scp.Trigger.TimeOut}");

                if (scp.Trigger.HasDelay)
                {
                    Console.WriteLine($"  Trigger delay max         : {scp.Trigger.DelayMax}");
                    Console.WriteLine($"  Trigger delay             : {scp.Trigger.Delay}");
                }

                if (scp.HasPresamplesValid)
                    Console.WriteLine($"  Presamples valid          : {scp.PresamplesValid}");
            }

            for (var ch = 0; ch < scp.Channels.Count; ch++)
            {
                OscilloscopeChannel channel = scp.Channels[ch];
                Console.WriteLine($"  Channel {ch + 1}:");
                Console.WriteLine($"    Connector type          : {channel.ConnectorType}");
                Console.WriteLine($"    Differential            : {channel.IsDifferential}");
                Console.WriteLine($"    Impedance               : {channel.Impedance}");
                Console.WriteLine($"    SureConnect             : {channel.HasSureConnect}");
                Console.WriteLine($"    Available               : {channel.IsAvailable}");
                Console.WriteLine($"    Enabled                 : {channel.Enabled}");
                Console.WriteLine($"    Bandwidths              : {string.Join(", ", channel.BandWidths)}");
                Console.WriteLine($"    Bandwidth               : {channel.BandWidth}");
                Console.WriteLine($"    Couplings               : {channel.Couplings}");
                Console.WriteLine($"    Coupling                : {channel.Coupling}");
                Console.WriteLine($"    Auto ranging            : {channel.AutoRanging}");
                Console.WriteLine($"    Ranges                  : {string.Join(", ", channel.Ranges)}");
                Console.WriteLine($"    Range                   : {channel.Range}");
                if (channel.HasSafeGround)
                {
                    Console.WriteLine($"    SafeGround enabled      : {channel.SafeGroundEnabled}");
                    Console.WriteLine($"    SafeGround threshold max: {channel.SafeGroundThresholdMax}");
                    Console.WriteLine($"    SafeGround threshold min: {channel.SafeGroundThresholdMin}");
                    Console.WriteLine($"    SafeGround threshold    : {channel.SafeGroundThreshold}");
                }

                if (channel.HasTrigger)
                {
                    var trigger = channel.Trigger!;
                    Console.WriteLine("    Trigger:");
                    Console.WriteLine($"      Available             : {trigger.IsAvailable}");
                    Console.WriteLine($"      Enabled               : {trigger.Enabled}");
                    Console.WriteLine($"      Kinds                 : {trigger.Kinds}");
                    Console.WriteLine($"      Kind                  : {trigger.Kind}");
                    Console.WriteLine($"      Level modes           : {trigger.LevelModes}");
                    Console.WriteLine($"      Level mode            : {trigger.LevelMode}");

                    if (trigger.Level.Count != 0)
                        Console.WriteLine($"      Levels                : {string.Join(", ", trigger.Level)}");

                    if (trigger.Hysteresis.Count != 0)
                        Console.WriteLine($"      Hysteresis            : {string.Join(", ", trigger.Hysteresis)}");

                    Console.WriteLine($"      Conditions            : {trigger.Conditions}");
                    if (trigger.Conditions != Constants.TIEPIE_HW_TCM_NONE)
                        Console.WriteLine($"      Condition             : {trigger.Condition}");

                    if (trigger.Time.Count != 0)
                        Console.WriteLine($"      Times                 : {string.Join(", ", trigger.Time)}");
                }
            }

            PrintTriggerInputsInfo(scp.TriggerInputs);
            PrintTriggerOutputsInfo(scp.TriggerOutputs);
        }
        catch (System.Exception e)
        {
            Console.WriteLine($"Exception: {e.Message}");
        }
    }

    public static void PrintGeneratorInfo(Generator gen)
    {
        try
        {
            Console.WriteLine("Generator:");
            Console.WriteLine($"  Connector type            : {gen.ConnectorType}");
            Console.WriteLine($"  Differential              : {gen.IsDifferential}");
            Console.WriteLine($"  Controllable              : {gen.IsControllable}");
            Console.WriteLine($"  Impedance                 : {gen.Impedance}");
            Console.WriteLine($"  Resolution                : {gen.Resolution}");
            Console.WriteLine($"  Output value min          : {gen.OutputValueMin}");
            Console.WriteLine($"  Output value max          : {gen.OutputValueMax}");
            Console.WriteLine($"  Output enable             : {gen.OutputEnable}");
            if (gen.HasOutputInvert)
                Console.WriteLine($"  Output invert             : {gen.OutputInvert}");

            Console.WriteLine($"  Modes native              : {gen.ModesNative}");
            Console.WriteLine($"  Modes                     : {gen.Modes}");
            if (gen.Modes != GeneratorMode.Unknown)
            {
                Console.WriteLine($"  Mode                      : " + gen.Mode);
                if (gen.Mode.HasFlag(GeneratorMode.BurstCount))
                {
                    Console.WriteLine($"  Burst active              : {gen.IsBurstActive}");
                    Console.WriteLine($"  Burst count max           : {gen.BurstCountMax}");
                    Console.WriteLine($"  Burst count               : {gen.BurstCount}");
                }
                if (gen.Mode.HasFlag(GeneratorMode.BurstSampleCount))
                {
                    Console.WriteLine($"  Burst sample count max    : {gen.BurstSampleCountMax}");
                    Console.WriteLine($"  Burst sample count        : {gen.BurstSampleCount}");
                }
                if (gen.Mode.HasFlag(GeneratorMode.BurstSegmentCount))
                {
                    Console.WriteLine($"  Burst segment count max    : {gen.BurstSegmentCountMax}");
                    Console.WriteLine($"  Burst segment count        : {gen.BurstSegmentCount}");
                }
            }

            Console.WriteLine($"  Signal types              : {gen.SignalTypes}");
            Console.WriteLine($"  Signal type               : {gen.SignalType}");

            if (gen.HasAmplitude)
            {
                Console.WriteLine($"  Amplitude min             : {gen.AmplitudeMin}");
                Console.WriteLine($"  Amplitude max             : {gen.AmplitudeMax}");
                Console.WriteLine($"  Amplitude                 : {gen.Amplitude}");
                Console.WriteLine($"  Amplitude Ranges          : {string.Join(", ", gen.AmplitudeRanges)}");
                Console.WriteLine($"  Amplitude range           : {gen.AmplitudeRange}");
                Console.WriteLine($"  Amplitude auto ranging    : {gen.AmplitudeAutoRanging}");
            }

            if (gen.HasFrequency)
            {
                Console.WriteLine($"  Frequency modes           : {gen.FrequencyModes}");
                Console.WriteLine($"  Frequency mode            : {gen.FrequencyMode}");
                Console.WriteLine($"  Frequency min             : {gen.FrequencyMin}");
                Console.WriteLine($"  Frequency max             : {gen.FrequencyMax}");
                Console.WriteLine($"  Frequency                 : {gen.Frequency}");
            }

            if (gen.HasOffset)
            {
                Console.WriteLine($"  Offset min                : {gen.OffsetMin}");
                Console.WriteLine($"  Offset max                : {gen.OffsetMax}");
                Console.WriteLine($"  Offset                    : {gen.Offset}");
            }

            if (gen.HasPhase)
            {
                Console.WriteLine($"  Phase min                 : {gen.PhaseMin}");
                Console.WriteLine($"  Phase max                 : {gen.PhaseMax}");
                Console.WriteLine($"  Phase                     : {gen.Phase}");
            }

            if (gen.HasSymmetry)
            {
                Console.WriteLine($"  Symmetry min              : {gen.SymmetryMin}");
                Console.WriteLine($"  Symmetry max              : {gen.SymmetryMax}");
                Console.WriteLine($"  Symmetry                  : {gen.Symmetry}");
            }

            if (gen.HasWidth)
            {
                Console.WriteLine($"  Width min                 : {gen.WidthMin}");
                Console.WriteLine($"  Width max                 : {gen.WidthMax}");
                Console.WriteLine($"  Width                     : {gen.Width}");
            }

            if (gen.HasEdgeTime)
            {
                Console.WriteLine($"  Leading edge time min     : {gen.LeadingEdgeTimeMax}");
                Console.WriteLine($"  Leading edge time max     : {gen.LeadingEdgeTimeMin}");
                Console.WriteLine($"  Leading edge time         : {gen.LeadingEdgeTime}");
                Console.WriteLine($"  Trailing edge time min    : {gen.TrailingEdgeTimeMax}");
                Console.WriteLine($"  Trailing edge time max    : {gen.TrailingEdgeTimeMin}");
                Console.WriteLine($"  Trailing edge time        : {gen.TrailingEdgeTime}");
            }

            if (gen.HasData)
            {
                Console.WriteLine($"  DataLength min            : {gen.DataLengthMin}");
                Console.WriteLine($"  DataLength max            : {gen.DataLengthMax}");
                Console.WriteLine($"  DataLength                : {gen.DataLength}");
            }

            PrintTriggerInputsInfo(gen.TriggerInputs);
            PrintTriggerOutputsInfo(gen.TriggerOutputs);
        }
        catch (System.Exception e)
        {
            Console.WriteLine($"Exception: {e.Message}");
        }
    }

    public static void PrintServerInfo(Server srv)
    {
        try
        {
            Console.WriteLine("Server:");
            Console.WriteLine($"  URL                       : {srv.Url}");
            Console.WriteLine($"  Name                      : {srv.Name}");
            Console.WriteLine($"  Description               : {srv.Description}");
            Console.WriteLine($"  IP address                : {srv.IpAddress}");
            Console.WriteLine($"  IP port                   : {srv.IpPort}");
            Console.WriteLine($"  Id                        : {srv.Id}");
            Console.WriteLine($"  Version                   : {srv.Version}");
            Console.WriteLine($"  Status                    : {srv.Status}");
            if (srv.LastError != ServerError.None)
                Console.WriteLine($"  Last error                : {srv.LastError}");
        }
        catch (System.Exception e)
        {
            Console.WriteLine($"Exception: {e.Message}");
        }
    }

    public static void PrintTriggerInputsInfo(TriggerInputs triggerInputs)
    {
        for (var i = 0; i < triggerInputs.Count; i++)
        {
            TriggerInput triggerInput = triggerInputs[i];
            Console.WriteLine($"  Trigger input {i}:");
            Console.WriteLine($"    Id                      : {triggerInput.Id}");
            Console.WriteLine($"    Name                    : {triggerInput.Name}");
            Console.WriteLine($"    Available               : {triggerInput.IsAvailable}");
            if (triggerInput.IsAvailable)
            {
                Console.WriteLine($"    Enabled                 : {triggerInput.Enabled}");
                Console.WriteLine($"    Kinds                   : {triggerInput.Kinds}");
                if (triggerInput.Kinds != Constants.TIEPIE_HW_TKM_NONE)
                    Console.WriteLine($"    Kind                    : {triggerInput.Kind}");
            }
        }
    }

    public static void PrintTriggerOutputsInfo(TriggerOutputs triggerOutputs)
    {
        for (var i = 0; i < triggerOutputs.Count; i++)
        {
            TriggerOutput triggerOutput = triggerOutputs[i];
            Console.WriteLine($"  TriggerOutput {i}:");
            Console.WriteLine($"    Id                      : {triggerOutput.Id}");
            Console.WriteLine($"    Name                    : {triggerOutput.Name}");
            Console.WriteLine($"    Enabled                 : {triggerOutput.Enabled}");
            Console.WriteLine($"    Events                  : {triggerOutput.Events}");
            Console.WriteLine($"    Event                   : {triggerOutput.Event}");
        }
    }
}
